/*  WXTide32 Harmonic tide clock and tide predictor
    Last modified 2002-02-10 by Mike Hopper for version 3.0

    This program uses the harmonic method to display tide levels
    either as a tide clock (showing the current level) or as a graph.
    All of the data and constants are read in from the harmonics file.
    Please refer to README for more information.

    Copyright (C) 1998  Michael Hopper
    Based on XTide.c Copyright (C) 1997  David Flater.
    ..and WinTide.c from 2/15/1996 by Paul C. Roberts
    Also starring:  Jef Poskanzer; Jack Greenbaum; Stan Uno; Dean Pentcheff.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


    The tide prediction algorithm used in this program was developed
    with United States Government funding, so no proprietary rights
    can be attached to it.  For more information, refer to the
    following publications:

    Manual of Harmonic Analysis and Prediction of Tides.  Special
    Publication No. 98, Revised (1940) Edition.  United States
    Government Printing Office, 1941.

    Computer Applications to Tides in the National Ocean Survey.
    Supplement to Manual of Harmonic Analysis and Prediction of Tides
    (Special Publication No. 98).  National Ocean Service, National
    Oceanic and Atmospheric Administration, U.S. Department of
    Commerce, January 1982.

    Parts of this program were carried over from XGrav, which was
    based on XSwarm, which was based on something else, ad infinitum.
*/

#include <windows.h>
#include <windowsx.h>  // <- nothing to do with Xwindows ;-) just some handy macros
#include <commdlg.h>
#include <shellapi.h>
#include <stdio.h>
#include <commctrl.h>
#include <winuser.h>
#include <mmsystem.h>

#include "everythi.h" // (Mostly) everything defined
#include "loclib.h"   // Station locator stuff
#include "wxtidres.h" // Windows resource definitions

#define WINWIDTH 175      /* default window width */ //mgh was 84
#define GRAPHWIDTH 640
#define SKINNYWIDTH 30
#define WINHEIGHT 300     /* default window height */

/* How much water at lowest tide; how much sky at highest tide. */
#define margin ((stagger && staggerfloat && lines && tinc && mark)? 0.11 : 0.05)
//#define margin 0.07 /* 0.05 */

#define BELLS_NAME "1bells.wav"

//BOOL DateTime_SetRange(HWND,DWORD,LPSYSTEMTIME);

/* External variables */

extern double next_ht_amplitude, next_lt_amplitude;
extern int    have_index;
extern char   IDX_type, IDX_zone[40], IDX_station_name[MAXNAMELEN], IDX_reference_name[MAXNAMELEN];
extern double IDX_lon, IDX_lat;
extern float  IDX_ht_mpy, IDX_ht_off, IDX_lt_mpy, IDX_lt_off;
extern short  IDX_rec_num, IDX_meridian, IDX_sta_num, IDX_ref_file_num;
extern short  IDX_ht_time_off, IDX_lt_time_off;
extern char   IDX_tzname[];
extern int    first_year, num_epochs, datemdy;
extern char  *tz_names[][2];

void set_local_tz();

/* Exported variables */
HINSTANCE g_hinst=NULL;
HWND      hwndMain=NULL;
COLORREF  fgmapdot_color,fgcurdot_color;
double Ilat, Ilon;
char   Izone[40], tadjust_last[256];
int    keep_index=0, new_params=0, auto_save=0, save_windows=1, overview = 0, have_sections = 0,
       had_user_offsets=0, sun_moon=0, num_days=5, num_months=1, increment_step=60, incremental_tides=0,
       caption = 0, textH = 11, nearestto=0, textascii=0, csv=0, map_cues=1, map_grid=1;
char   indexfile_name[MAXARGLEN]; // Index file name
char   userfile_name[MAXARGLEN];  // User Index file name
char   *fgmapdot_color_arg=NULL;  // Color of dots on station locator map
char   *fgcurdot_color_arg=NULL;  // Color of dots on station locator map
char   *youwant = NULL;
char   *custom_name = NULL;
char   *bmp_name = NULL;
int    have_BOGUS = 0, convert_BOGUS = 0;
RECT   WinDimMain = {1,1,100,100},
       WinDimText = {1,1,0,0},
       WinDimMap  = {1,1,0,0};

/* Local variables */
#define DISPLAY_TEXT 1
#define DISPLAY_GRAPH 2
#define TIMER_VALUE 1000

static char *szAccum=NULL, *output_filename=NULL;
static int  usescrollbar=0, notnowtime = 0;//, nLenAccum=0;
static BOOL done;
static COLORREF fgrise_color, fgfall_color, fgtext_color,
   fgmark_color, fgmllw_color, fgmiddle_color, bgday_color, bgnite_color;
static HPEN fgtext_color_gc,fgmark_color_gc,fgmiddle_color_gc,fgmllw_color_gc,
            fgrise_color_gc,fgfall_color_gc,fgmapdot_color_gc,fgcurdot_color_gc,
            bgday_color_gc, bgnite_color_gc;
static char *bgday_color_arg = NULL, *bgnite_color_arg = NULL;
static int winX, winY, testmode = 0, testspeed = 0, rising = -1,
  depthlines=0, stagger=0, staggerfloat=0, onedate=0, resetdate=0,
  hairycalibrate = 0, tmax, tmin, /*norename = 0,*/
  appendWXTide32 = 0, OnESCape = 0, UseIcon = 0, IconStyle=0, OnlyOne=0, WindowWasMax=0, ChangingWindow=0,
  cant_open_config_file=0;
static int hold_size_min_max=0, hold_move_min_max=0, hold_restore=0, haveSounds=0, ShipBells=0;
static int event_type, showampl=0, top_text_bottom, show_moons=0, ForceUpdateIcon=0, contextMenuDisable=0;
static unsigned int winW, winH;
static HWND hwndText=NULL,popup=NULL,scrollbar=NULL,nowbutton=NULL;
static HINSTANCE t_hinst;
static HDC display=NULL;
static HFONT gfont=NULL, tfont=NULL, bfont=NULL;
static double prev_tide, this_tide;
time_t gstart_time=0, window_left_time;
static int nowarn = 0, have_new_custom = 0, have_params = 0, hold_main_window = TRUE;
HFILE hf;
char szProgramName[256], szProgramPath[256], szAlternatePath[256], szConfigPath[256];
char HelpFileName[256];
OPENFILENAME ofn;
static FILE *textLFN=NULL;
static int  gFontSize= 8;
static char gFontName[LF_FACESIZE+1]="MS Sans Serif";
static int  tFontSize= 10;
static char tFontName[LF_FACESIZE+1]="Courier";
static char bellsname[] = BELLS_NAME;
static double tide2wl (double tide);
static char MRU[10][100];
static int max_mru=9;
HICON  wxtide32icon=NULL, trayicon=NULL;
NOTIFYICONDATA trayIcon;      // Taskbar tray icon
BOOL   inTray = FALSE;        // Shown in system tray ?

#define MODE_HAIRY    0
#define MODE_GRAPH    1
#define MODE_CLOCK    2
#define MODE_OVERVIEW 3
#define MAX_SECTION   3

typedef struct {
   char  *name;
   int   noampm;
   int   caption;
   int   hairycalibrate;
   int   depthlines;
   int   tstep;
   int   hinc;
   int   mark;
   float marklev;
   int   middle;
   int   mllw;
   int   show_moons;
   int   linegraph;
   int   lines;
   int   now;
   int   onedate;
   int   usescrollbar;
   int   showampl;
   int   skinny;
   int   stagger;
   int   staggerfloat;
   int   tinc;
   int   toplines;
   int   weekday;
//   int   utc;
   RECT  WinDimMain;
} section_entry;

static section_entry section[MAX_SECTION+1] = {
{"HAIRY",   0,0,1,0, 180,1,0,0.0, 0,0,1,0,1,1,0,0,1,0,0,0,1,0,0, 1,1,100,100},
{"GRAPH",   0,0,0,1, 180,1,0,0.0, 0,0,1,0,1,1,1,1,1,1,1,0,1,0,0, 1,1,100,100},
{"CLOCK",   0,0,0,0, 180,1,0,0.0, 0,0,1,0,1,1,0,0,1,0,0,0,0,0,0, 1,1,100,100},
{"OVERVIEW",0,0,0,0,1200,1,0,0.0, 0,0,1,1,1,0,1,1,1,2,1,1,1,0,0, 1,1,100,100} };
int section_number;

LRESULT CALLBACK TidesWndProc(HWND,UINT,WPARAM,LPARAM);
BOOL    CALLBACK TextTidesDlgProc(HWND hwndDlg,UINT msg,WPARAM wParam,LPARAM lParam);
LRESULT CALLBACK TidesMapProc(HWND,UINT,WPARAM,LPARAM);
static  void     NameWindow( HWND hwnd, char * name );
void    WinTideCheckMenus(HWND hwnd);
BOOL    SaveBitmap(char *name,HBITMAP bitmap);
HBITMAP CaptureWindow(HWND hwndWindow, int width, int heigth);
void    RestoreMainWindow(HWND);

#define MoveTo(hdc,x,y) MoveToEx(hdc,x,y,NULL)
#define SetWindowOrg(a,b,c) SetWindowOrgEx((a),(b),(c),NULL)
#define SetWindowExt(a,b,c) SetWindowExtEx((a),(b),(c),NULL)
#define SetViewportExt(a,b,c) SetViewportExtEx((a),(b),(c),NULL)

DWORD GetTextExtent(LPSTR text, int cbLen) {
    DWORD dwReturn;
    SIZE sz;

    GetTextExtentPoint32(display, text, cbLen, &sz);
    dwReturn = MAKELONG(sz.cx, sz.cy);

    return dwReturn;
}

//
//some Xfunctions can be replaced by inline fns
//
void XFillRectangle (HDC display, COLORREF color, int x, int y,int  w,int h) {
     RECT rc;
     HBRUSH hbr;
     rc.left=x;
     rc.top=y;
     rc.right=x+w;
     rc.bottom=y+h;
     hbr=CreateSolidBrush(color);

     FillRect(display,&rc,hbr);
     DeleteBrush(hbr);
}

void XDrawLine (HDC display, HPEN hpen, int x1,int y1,int x2,int y2) {
     hpen=SelectPen(display,hpen);
     MoveTo(display,x1,y1);
     LineTo(display,x2,y2);
     SelectPen(display,hpen);
}

// colour is totally different under windows. most of the
// probs under X don't arise. You never get denied a color
// (if it isn't in the colormap (palette in windows) you just get
// given the nearest.

///* Used by GetColor. */

///* If a color cannot be allocated, the following function assumes a
//   256-color colormap and searches for the closest match. */

static long GetColor (char *color) {
  int r,g,b;

  if (color) {
    if (sscanf (color, "rgb:%x/%x/%x", &r, &g, &b) != 3)
      barf (BADCOLORSPEC);
  }
  if( ! ( (r >= 0 && r <= 255) &&
          (g >= 0 && g <= 255) &&
          (b >= 0 && b <= 255) ) )
    barf (BADCOLORSPEC);

  return RGB(r,g,b);
}

// don't know how relevant this is lose all the X calls for now
static void die () {
  DeletePen(fgrise_color_gc);
  DeletePen(fgfall_color_gc);
  DeletePen(bgday_color_gc);
  DeletePen(bgnite_color_gc);
  DeletePen(fgtext_color_gc);
  DeletePen(fgmark_color_gc);
  DeletePen(fgmiddle_color_gc);
  DeletePen(fgmllw_color_gc);
  DeleteFont(gfont);
}


/*-----------------10/2/2002 6:59AM-----------------
* Add station name to Most Recently Used list
* * --------------------------------------------------*/
void add_mru( char *station ) {
  char name[256], *p, found_c;
  int i, found;

  strcpy(name, "0 ");
  strcat(name, station);
  if (strlen(name) >= sizeof(MRU[0]) - 1)
      name[sizeof(MRU[0])-1] = '\0';

  found = -1;
  for (i=0; i<10; i++) {
    p = MRU[i];
    if (isdigit(*p) && !stricmp(name+1, p+1)) found = i;
  }

  if (found == -1) { // This is a new entry
    for (i=0; i<10; i++) {
      p = MRU[i];
      if (isdigit(*p)) *p += 1;
      if (found==-1 && !isdigit(*p)) {
        strcpy(p, name);
        found = 0;
      }
    }
  }

  else { // This is an existing entry
    found_c = MRU[found][0];
    for (i=0; i<10; i++) {
      p = MRU[i];
      if (i == found)
          strcpy(p, name);
      else if (isdigit(*p) && *p < found_c )
          *p += 1;
    }
  }
}

/*-----------------10/2/2002 7:29AM-----------------
* Get MRU entry, called by write config and ID_MRU#
* mru is 0..9
* --------------------------------------------------*/
char *get_mru_entry( int mru ) {
  char *p, mru_id;
  int i;

  if (mru > max_mru) return( 0 );
  mru_id = mru + '0';
  for (i=0; i<10; i++) {
    p = MRU[i];
    if (*p == mru_id)
      return( p );
  }
  return( 0 );
}

/*-----------------10/2/2002 7:29AM-----------------
* Remove MRU entry, called by Custom station when deleting
* --------------------------------------------------*/
void remove_mru( char *name ) {
  char *p, found_c, t_name[100];
  int i, found;

  strcpy(t_name, name);
  if (t_name[strlen(t_name)-4] == ' ' &&
      t_name[strlen(t_name)-3] == '(' &&
      t_name[strlen(t_name)-1] == ')'    )
      t_name[strlen(t_name)-4] = '\0';
  found = -1;
  for (i=0; i<10; i++) {
    p = MRU[i];
    if (isdigit(*p) && !stricmp(t_name, p+2)) found = i;
  }
  if (found >= 0) {
    new_params = TRUE;
    found_c = MRU[found][0];
    for (i=0; i<10; i++) {
      p = MRU[i];
      if (i == found)
        *p = ' ';
      else if (isdigit(*p) && *p > found_c )
        *p -= 1;
    }
  }
}

/*-----------------10/2/2002 7:29AM-----------------
* Load MRU entry, called by recent station logic
* mru is 0..9
* --------------------------------------------------*/
void load_mru( int mru ) {
  char *p;
  p = get_mru_entry(mru);
  if ( p ) {
//    if (!have_index)
//      init_index_file(TRUE, display); // Load full index
    if (!load_location_data(p+2, 0)) {
      char s[256];
      sprintf(s,"Can't find %s.", p+2);
      MessageBox(hwndMain, s, "MRU Station not found", MB_OK|MB_ICONWARNING);
      remove_mru(p+2);
    }
  }
  NameWindow(hwndMain, custom_name);
  WinTideCheckMenus(hwndMain);
}


void draw_moon(int moonX, int phase) {
HDC moonDC, phaseDC;
HBITMAP hMask, hPhase, hBmpOld;
BITMAP bm;
COLORREF oldbg,oldfg;
int maskW, maskH, moonDCX, moonDCY;
DWORD maskOP;
char *szMoonMask="MOONMASK",*szMoonPhase;

   display = display;
   hMask = LoadBitmap(g_hinst, szMoonMask);
   GetObject(hMask, sizeof(BITMAP), &bm);
   moonDC=CreateCompatibleDC(display);
   hBmpOld=SelectObject(moonDC, hMask);

   oldbg = SetBkColor(  display, RGB(255,255,255)); // Background is WHITE
   oldfg = SetTextColor(display, RGB(  0,  0,  0)); // Black dots are BLACK
   maskW = bm.bmWidth;
   moonDCX = moonX-(maskW/2);
   maskH = bm.bmHeight;
   moonDCY = top_text_bottom;
   if ((moonDCX > -maskW) && (moonDCX < winW) &&
       (moonDCY > 0) && ((moonDCY+maskH) < winH)) {
      if (phase == 2) { // Full moon
// Full moon, Use invert of mask to make a white area on the screen.
      BitBlt(display,moonDCX,moonDCY,maskW,maskH,moonDC,0,0,MERGEPAINT);
      }
      else {
// First use mask to make a black area on the screen.
      BitBlt(display,moonDCX,moonDCY,maskW,maskH,moonDC,0,0,SRCAND);

      if (phase == 0) { // New moon
         szMoonPhase = "MOONNEW";
         maskOP      = SRCERASE;
      }
      else if ((phase == 1 && IDX_lat > 0.0) ||    // First quarter moon in north OR
               (phase == 3 && IDX_lat < 0.0)   ) { // third quarter moon in south
         szMoonPhase = "MOONHALF";                 // illuminates right half
         maskOP      = SRCERASE;
      }
      else  { // Third quarter moon
         szMoonPhase = "MOONHALF";
         maskOP      = NOTSRCERASE;
      }
      hPhase = LoadBitmap(g_hinst, szMoonPhase);
      GetObject(hPhase, sizeof(BITMAP), &bm);
      phaseDC=CreateCompatibleDC(display);
      SelectObject(phaseDC, hPhase);
// Set up masking template
      BitBlt(moonDC, 0,0,maskW,maskH,phaseDC,0,0,maskOP);
// Now merge the template with the display
      BitBlt(display,moonDCX,moonDCY,maskW,maskH,moonDC,0,0,SRCPAINT);

      DeleteDC(phaseDC);
      DeleteObject(hPhase);
      }
   }
   SelectObject(moonDC, hBmpOld);
   DeleteDC(moonDC);
   SetBkColor(  display, oldbg);
   SetTextColor(display, oldfg);
   DeleteObject(hMask);
}

static void draw_moons() {
time_t moon_time;
int moon;
   if (show_moons) {
      moon_time = window_left_time - (DAYSECONDS / 2); // Start back 1/2 day
      moon=next_moon_phase(&moon_time, 1);
      while (moon_time < (window_left_time + (tstep*winW))) {
         draw_moon((moon_time-window_left_time)/tstep, moon);
         moon=next_moon_phase(&moon_time, 1);
      }
   }
}


static void close_popup() {
  if (popup) {
     DestroyWindow(popup);
     popup = NULL;
  }
}

static void open_popup(DWORD mouseXY, char *text) {
int mouseX, mouseY, cxscreen, startX, startY, side; /*cyscreen*/
PAINTSTRUCT ps;
RECT purc = {0,0,0,0};
HFONT oldfont;
#define POPUP_BORDER 1

  close_popup();
// Open window with dummy position and size to get device context
  popup = CreateWindow("STATIC", "", WS_POPUP|WS_VISIBLE,//|WS_BORDER,
                       10,10,250,50, hwndMain, NULL, t_hinst, NULL);

  BeginPaint(popup,&ps);
  oldfont = SelectFont(ps.hdc,gfont);
// Get extents for this text
  DrawText(ps.hdc, text, strlen(text), &purc, DT_CALCRECT);
  purc.right  += POPUP_BORDER*2;
  purc.bottom += POPUP_BORDER*2;

  mouseX = LOWORD(mouseXY);
  mouseY = HIWORD(mouseXY);
  cxscreen = GetSystemMetrics(SM_CXSCREEN);
//  cyscreen = GetSystemMetrics(SM_CYFULLSCREEN);
  if (side=(cxscreen > (mouseX + winX + purc.right)))
       startX = mouseX + winX;
  else startX = mouseX + winX - purc.right;
  if (0 < (mouseY + winY - purc.bottom))
       startY = mouseY + winY - purc.bottom;
  else startY = mouseY + winY;
// Now make window the right size and in the right place
  MoveWindow(popup,startX,startY,purc.right,purc.bottom,FALSE);
//  SetTextColor(ps.hdc,fgtext_color);
//  SetBkColor(ps.hdc,RGB(255,255,255));
  FillRect( ps.hdc, &purc, GetStockObject(WHITE_BRUSH));
  FrameRect(ps.hdc, &purc, GetStockObject(BLACK_BRUSH));
  purc.left   += POPUP_BORDER;
  purc.top    += POPUP_BORDER;
  purc.right  -= POPUP_BORDER;
  purc.bottom -= POPUP_BORDER;
  DrawText(ps.hdc,text,strlen(text),&purc,
           DT_WORDBREAK|(side? DT_LEFT : DT_RIGHT));
  SelectFont(ps.hdc,oldfont);
  EndPaint(popup,&ps);
}


time_t CrosshairTime=0;
int CrosshairEnable = FALSE, CrosshairDrawn=0, UseCrosshairPopup=1;

static void DrawCrosshair(time_t CrossTime, HDC h) {
int ropsave, CrosshairX, CrosshairY;
HDC hdc;
  if (CrossTime > 0) {
    if (h) hdc = h;
    else   hdc = GetDC(hwndMain);
    ropsave= SetROP2(hdc, R2_NOT);
    CrosshairX = (CrossTime-window_left_time)/tstep;
    CrosshairY = (int)tide2wl ((time2asecondary(CrossTime) - fakedatum) / fakeamplitude);
    MoveTo(hdc, CrosshairX, 0);
    LineTo(hdc, CrosshairX, winH);
    MoveTo(hdc, 0,    CrosshairY);
    LineTo(hdc, winW, CrosshairY);
    SetROP2(hdc, ropsave);
    if (!h) ReleaseDC(hwndMain, hdc);
    if (UseCrosshairPopup && CrosshairDrawn == 0) {
      char s[100], stime[64];
      int skinny_save = skinny;
      skinny=1;
      do_timestamp( stime, tmtime(CrossTime) );
      sprintf(s,"%s %1.2lf%s",stime, time2asecondary(CrossTime), units_abbrv);
      open_popup(MAKELONG(CrosshairX,0), s);
      skinny=skinny_save;
      SetFocus(hwndMain);
    }
    CrosshairDrawn = CrossTime;
  }
}

static void UndrawCrosshair() {
  if (CrosshairEnable && CrosshairDrawn > 0) {
    if (UseCrosshairPopup) close_popup();
    DrawCrosshair( CrosshairDrawn, NULL );
  }
  CrosshairDrawn = 0;
//  UseCrosshairPopup = 0;
}

static void DoCrosshair(HDC hdc) {
  if (UseCrosshairPopup && CrosshairDrawn != 0)
    close_popup();
  CrosshairDrawn = 0;
  if (CrosshairTime <= 0) {
    CrosshairTime = time( NULL );
  }
  else if (CrosshairTime <= window_left_time) {
    CrosshairTime = window_left_time + HOURSECONDS;
    CrosshairEnable = FALSE;
  }
  else if (CrosshairTime >= window_left_time + winW*tstep) {
    CrosshairTime = window_left_time + winW*tstep - HOURSECONDS;
    CrosshairEnable = FALSE;
  }
  if (CrosshairEnable)
    DrawCrosshair(CrosshairTime, hdc);
  else {
    CrosshairDrawn = 0;
//    UseCrosshairPopup = 0;
  }
}

static char *astro_info(time_t mousetime) {
static char s[512];
char stime[64],sdate[64],smdate[64],s_sun[80],s_moon[80];
time_t moon_time;
int moon, now_day, skinny_save = skinny;
struct tm *tmmoon;
char *phase_names[4] = {
  {"New moon, first quarter"},
  {"First quarter moon, full"},
  {"Full moon, last quarter"},
  {"Last quarter moon, new"} };

  skinny = 0;
  do_timestamp( stime, tmtime(mousetime) );
  do_datestamp( sdate, tmtime(mousetime) );

  s_sunrise_set(s_sun, mousetime);
  s_moonrise_set(s_moon, mousetime);

  moon_time = mousetime;
  moon = next_moon_phase(&moon_time, -1);
  next_moon_phase(&moon_time, 1);
  now_day = tmtime(mousetime)->tm_mday;
  tmmoon = tmtime(moon_time);
  if (now_day == tmmoon->tm_mday) { // It's today!
    strcpy(smdate, "at ");
    do_timestamp( smdate+strlen(smdate), tmmoon);
  }
  else {
    strcpy(smdate, "on ");
    do_datestamp( smdate+strlen(smdate), tmmoon);
  }
  sprintf(s,"%s  %s  %01.2lf %s\r\n"
          "%s\r\n%s\r\n%s %s",
          stime, sdate, time2asecondary(mousetime), units,
          s_sun, s_moon, phase_names[moon], smdate);
  skinny = skinny_save;
  return(s);
}

#define SCROLL_DAY_RANGE 30
#define SCROLL_MON_RANGE 12
static int    scroll_range, scroll_pos;
static time_t scroll_start_time;

static time_t scroll_bar_2_time_t(int pos) {
struct tm *tm;
int offset = pos - scroll_pos;

  tm = tzlocaltime( &scroll_start_time );
  tm->tm_hour = tm->tm_min = tm->tm_sec = 0;
  if (abs(offset) <= SCROLL_DAY_RANGE)
    tm->tm_mday += offset;
  else if (abs(offset) <= (SCROLL_DAY_RANGE + SCROLL_MON_RANGE)) {
    if (offset < 0)
         tm->tm_mon += (offset + SCROLL_DAY_RANGE);
    else tm->tm_mon += (offset - SCROLL_DAY_RANGE);
  }
  else {
    if (offset < 0)
         tm->tm_year += (offset + (SCROLL_DAY_RANGE + SCROLL_MON_RANGE));
    else tm->tm_year += (offset - (SCROLL_DAY_RANGE + SCROLL_MON_RANGE));
  }
  return( mktime( tm ) );
}

static void DoScrollBar(int adj) {
int  fmoff, lmoff, rpad, lpad, nowW, nowH = GetSystemMetrics(SM_CYHSCROLL);
time_t first_day, last_day, now_day, now_time;
struct tm *tm;
RECT clientRC;

  if (scrollbar) DestroyWindow(scrollbar);
  if (nowbutton) DestroyWindow(nowbutton);
  scrollbar = nowbutton = NULL;

//  nowW = notnowtime? 35 : 0;
  nowW = 35;
  if (usescrollbar && graphmode) {
    if (adj) winH -= nowH;
    now_time = time(NULL);
    tm = tzlocaltime( &now_time ); // Just to set the pointer!
//Save first date in data set
    tm->tm_year = first_year - 1900;
    tm->tm_mon = tm->tm_hour = tm->tm_min = tm->tm_sec = 0;
    tm->tm_mday = 3;
    first_day = mktime( tm ) / DAYSECONDS;
//Save last date in data set
    tm->tm_year = first_year+num_epochs-1 - 1900;
    tm->tm_mon = tm->tm_hour = tm->tm_min = tm->tm_sec = 0;
    tm->tm_mon = 11;
    tm->tm_mday = 29;
    last_day = mktime( tm ) / DAYSECONDS;
//Get current (or last forced) date
    scroll_start_time = gstart_time? gstart_time : time(NULL);
    tm = tzlocaltime( &scroll_start_time );
    now_day = scroll_start_time / DAYSECONDS;
//Find months and padding needed from start of data to now
    fmoff=(tm->tm_year+1900 - first_year)*12 + tm->tm_mon;
    if (fmoff >= SCROLL_MON_RANGE)
         lpad = SCROLL_MON_RANGE;
    else lpad = fmoff;
//Find months and padding needed from now to end of data
    lmoff=(first_year+num_epochs-1 - tm->tm_year-1900)*12 + (11- tm->tm_mon);
    if (lmoff >= SCROLL_MON_RANGE)
         rpad = SCROLL_MON_RANGE;
    else rpad = lmoff;
//Adjust for days before/after
    if ((now_day - first_day) >= SCROLL_DAY_RANGE)
         lpad += SCROLL_DAY_RANGE;
    else lpad += (now_day - first_day);

    if ((last_day - now_day) >= SCROLL_DAY_RANGE)
         rpad += SCROLL_DAY_RANGE;
    else rpad += (last_day - now_day);

    scroll_range = num_epochs-1 + lpad + rpad;
    scroll_pos = tm->tm_year+1900 - first_year + lpad;
//Do it!
    GetClientRect(hwndMain,&clientRC);
    scrollbar=CreateWindow("SCROLLBAR", "", WS_CHILD|WS_VISIBLE|SBS_HORZ|SBS_BOTTOMALIGN,
                            nowW, 0, clientRC.right-nowW, clientRC.bottom, hwndMain, 0, g_hinst, NULL);
    SetScrollRange(scrollbar,SB_CTL,0,scroll_range,FALSE);
    SetScrollPos(scrollbar,SB_CTL,scroll_pos,TRUE);
    nowbutton =
      CreateWindow("BUTTON", "Now", WS_CHILD|WS_VISIBLE|BS_PUSHBUTTON,
                   0, clientRC.bottom-nowH, nowW, nowH, hwndMain, (HMENU)ID_GONOW, g_hinst, NULL);
    if (bfont == NULL) {
      HDC     hDCnow = GetDC(nowbutton);
      LOGFONT lf;
      memset(&lf, 0, sizeof(lf));
      lf.lfHeight = -MulDiv(8, GetDeviceCaps(hDCnow, LOGPIXELSY), 72);
      strcpy(lf.lfFaceName,"MS Sans Serif");
      bfont=CreateFontIndirect(&lf);
      ReleaseDC(hwndMain,hDCnow);
    }
    SendMessage(nowbutton,WM_SETFONT,(WPARAM)bfont, MAKELONG(TRUE,0));
    if (!notnowtime)
      SetWindowText(nowbutton, "Goto");
   }
  else if (adj) winH += nowH;
}

/*-----------------12/16/2005 8:19AM----------------
 * Change to new main graphics mode
 * Windows can't handle min'ed/max'ed windows with
 * different base sizes.
 * When we have to change modes, we have to sequence the operations:
 * 1: Read new mode values (including new window params)
 * 2: Restore main window normal (not min or max) size
 * 3: Read new mode window limits again and move/size window
 * 4: Restore new mode window to minimum/maximum
 * --------------------------------------------------*/
void ChangeMainWindowMode(HWND hwnd) {
  if (IsIconic(hwnd) || IsZoomed(hwnd)) {
    if (IsIconic(hwnd))
         ChangingWindow = (WindowWasMax? -2 : -1);
    else ChangingWindow =  1;
    hold_restore = 1;
    ShowWindow(hwnd, SW_SHOWNORMAL);
    PostMessage(hwnd, WM_COMMAND, ID_CHANGE_WINDOW, 0);
  }
  else
    ChangingWindow = 0;
}

/*-----------------12/15/2005 6:20PM----------------
 * Restore main graphics window if it is minimized
 * --------------------------------------------------*/
void RestoreMainWindow(HWND hwnd) {
  if (IsIconic(hwnd)) {
//    hold_restore = 1;
//    ShowWindow(hwnd, SW_SHOW);
    if (WindowWasMax)
         ShowWindow(hwnd, SW_SHOWMAXIMIZED);
    else ShowWindow(hwnd, SW_SHOWNORMAL);
//    InvalidateRect(hwnd,NULL,TRUE);
//    UpdateWindow(hwnd);
//    hold_restore = 1;
  }
}


static void Init_Application(HINSTANCE hinst) {
  WNDCLASS wc;
  INITCOMMONCONTROLSEX cc;

  // Needed for month calendar control
  cc.dwSize = sizeof(INITCOMMONCONTROLSEX);
  cc.dwICC  = ICC_DATE_CLASSES;
//#ifdef InitCommonControlsEx
  InitCommonControlsEx(&cc);            // comctl32.dll version 4.70 or later
//#endif

  wc.style         = 0;//CS_HREDRAW|CS_VREDRAW;
  wc.lpfnWndProc   = TidesWndProc;
  wc.cbClsExtra    = 0;
  wc.cbWndExtra    = 0;
  wc.hInstance     = hinst;
  wc.hIcon         = wxtide32icon = trayicon = LoadIcon(hinst, "WXTide");
  wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = NULL; //CreateSolidBrush(bgday_color);
  wc.lpszMenuName  = "WXTide";
  wc.lpszClassName = "WXTide";

  RegisterClass(&wc);
}

static void NameWindow( HWND hwnd, char * name ) {
static char *PROG_NAME = {" - WXTide32"};
char *win_name;
  if ( appendWXTide32 ) {
    win_name = malloc(strlen(name) + strlen(PROG_NAME) + 1);
    strncpy(win_name, name, strlen(name)+1);
    strcat(win_name, PROG_NAME);
    SetWindowText(hwnd, win_name);
    free(win_name);
  }
  else
    SetWindowText(hwnd, name);
}

/*-----------------12/27/2002 4:58PM----------------
 * Validate window dimensions
 * --------------------------------------------------*/
static void ValidateWindowDimensions(RECT *win) {
RECT work;

  SystemParametersInfo(SPI_GETWORKAREA, 0, &work, 0);

  if ((win->right - win->left) < SKINNYWIDTH)
    win->right = win->left + SKINNYWIDTH;
  if (win->left < work.left)
    win->left   = work.left;
  if (win->right > work.right)
    win->right = work.right;
  if ((win->right - win->left) < SKINNYWIDTH) {
    win->left  = work.left;
    win->right = work.right;
  }

  if ((win->bottom - win->top) < 30)
    win->bottom = win->top + 30;
  if (win->top < work.top)
    win->top    = work.top;
  if (win->bottom > work.bottom)
    win->bottom = work.bottom;
  if ((win->bottom - win->top) < 30) {
    win->top    = work.top;
    win->bottom = work.bottom;
  }

}
/* Create and setup the window. */
static void Create_Window (/*char *geometry*/HINSTANCE hinst) {

  ValidateWindowDimensions(&WinDimMain);
  winX = WinDimMain.left;
  winY = WinDimMain.top;
  winW = WinDimMain.right  - WinDimMain.left;
  winH = WinDimMain.bottom - WinDimMain.top;

  hwndMain=CreateWindow("WXTide", "", WS_OVERLAPPEDWINDOW|WS_CLIPCHILDREN,
                winX, winY, winW, winH, NULL, 0, hinst, NULL);

//#define GRAPH_TITLE " Graph"
//  {
//    char *win_name;

//    if (graphmode) {
//      win_name = malloc(strlen(custom_name) + strlen(GRAPH_TITLE) + 1);
//      strncpy(win_name, custom_name, strlen(custom_name)+1);
//      strcat(win_name, GRAPH_TITLE);
//    }
//    else win_name = custom_name;
//    else win_name = custom_name;

//    NameWindow(hwndMain, win_name);

//    if (graphmode) free(win_name);
//  }
  NameWindow(hwndMain, custom_name);
}

/* Convert a normalized tide to a Y coordinate in the window. */
static double
tide2wl (double tide)
{
  return (((double)winH * (1.0 - 2.0 * margin)) * (-tide) * 0.5
         + (double)winH * 0.5);
}

/* Inverse. */
static double
wl2tide (int y)
{
  return -2.0*(y - winH * 0.5) / ((double)winH * (1.0 - 2.0 * margin));
}

/* Text is centered around x.  Will break horribly if the font changes.
Most of this function has to do with making the window wider if the text didn't fit. */
static void
center_text (int x, int y, char *text)
{
  RECT r;
  int l = strlen (text);
  DWORD dwExtents;
  r.top = r.left=0;
  r.bottom=winH;
  r.right=winW;
  dwExtents=GetTextExtent(text,l);
  ExtTextOut(display,x-(LOWORD(dwExtents)>>1),y-HIWORD(dwExtents)+1,ETO_CLIPPED,&r,text,l,NULL);
}

void draw_caption() {
  if (caption)
     center_text (winW>>1, textH, custom_name);
}
/*-----------------12/23/2002 4:15PM----------------
 *
 * --------------------------------------------------*/
int get_tide_day(time_t t) {
struct tm ttm;
  ttm = *(tzlocaltime (&t));
  return(ttm.tm_mday);
}
/* Draw text telling when the high tide is.  Now kluged to do low tide as
   well, and currents, and everything.... */
static void
write_high_tide (int x, time_t tide_time)
{
  char temp[80], next_hta_text[80], next_lta_text[80], now_text[80];
  int  bottom = winH-1, y;
  char longest[80], mediumest[80], units[80];

  if (strcmp (units_abbrv, "unknown") && (skinny != 2))
       strcpy(units, units_abbrv);
  else strcpy(units, "");

  do_timestamp (now_text, tmtime (time (NULL)));
  if (showampl) {
    sprintf(next_hta_text, "%s %01.1f%s", next_ht_text, next_ht_amplitude, units);
    sprintf(next_lta_text, "%s %01.1f%s", next_lt_text, next_lt_amplitude, units);
    sprintf(now_text+strlen(now_text)," %01.1f%s",time2asecondary(time(NULL)),units);
  }
  else {
    strcpy(next_hta_text, next_ht_text);
    strcpy(next_lta_text, next_lt_text);
  }

  top_text_bottom = caption? textH : 0;
  if (graphmode) {
//    center_text (x, 12, next_ht_date);
    if (event_type &1) {                   //mgh added 1 = low tide
       if (showampl)
         sprintf(next_hta_text,"%s %01.1f%s", next_ht_text, next_lt_amplitude, units);
       else strcpy(next_hta_text, next_ht_text);
    }

    if (!onedate || resetdate != get_tide_day(tide_time)) {
      resetdate = get_tide_day(tide_time);
      if (stagger && !staggerfloat && event_type & 2) // 2 = High tide
        center_text (x, top_text_bottom+textH*2, next_ht_date);
      else
        center_text (x, top_text_bottom+textH, next_ht_date);
    }

    if (stagger && !staggerfloat) {
//      top_text_bottom = 22;
      if (event_type &1)
        center_text (x, top_text_bottom+textH*2, next_hta_text);
      else
        center_text (x, top_text_bottom+textH, next_hta_text);
      top_text_bottom += textH*2;
    }
    else if (stagger && staggerfloat) {
//      top_text_bottom = 12;
      y = (int)tide2wl (time2secondary( tide_time ));
      if (event_type &1)                        // low tide
        center_text (x, y+1+textH, next_hta_text);
      else                                      // high tide
        center_text (x, y,  next_hta_text);
      top_text_bottom += textH;
    }
    else {
      top_text_bottom += textH*2;
      center_text (x, top_text_bottom, next_hta_text);
    }
    return;
  }

  if (hairy && lines) {
     bottom -= 9;
     if (tinc) bottom -= textH;
  }

  sprintf(longest,  "Next High Tide %s", next_hta_text);
  sprintf(mediumest,"Flood %s",          next_hta_text);

  if (winW >= 6+(LOWORD(GetTextExtent(longest, lstrlen(longest))))) {
// Longest fits on one line
    top_text_bottom += textH;
    if (iscurrent) {
       sprintf (temp, "Next Max Flood %s", next_hta_text);
       center_text (x, top_text_bottom, temp);
       sprintf (temp, "Next Max Ebb %s",   next_lta_text);
       center_text (x, bottom, temp);
//       top_text_bottom = 12;
    }
    else {
       sprintf (temp, "Next High Tide %s", next_hta_text);
       center_text (x, top_text_bottom, temp);
       sprintf (temp, "Next Low Tide %s",  next_lta_text);
       center_text (x, bottom, temp);
//       top_text_bottom = 12;
    }
    if (now) {
       top_text_bottom += textH;
       sprintf(temp, "Time Now %s", now_text);
       center_text (x, top_text_bottom, temp);
//       top_text_bottom = 23;
    }
  }

  else if (winW >= 6+(LOWORD(GetTextExtent(mediumest, lstrlen(mediumest))))) {
// Medium line fits
    top_text_bottom += textH;
    if (iscurrent) {
       sprintf (temp, "Flood %s", next_hta_text);
       center_text (x, top_text_bottom, temp);
       sprintf (temp, "Ebb %s",   next_lta_text);
       center_text (x, bottom, temp);
//       top_text_bottom = 12;
    }
    else {
       sprintf (temp, "High %s", next_hta_text);
       center_text (x, top_text_bottom, temp);
       sprintf (temp, "Low %s",  next_lta_text);
       center_text (x, bottom, temp);
//       top_text_bottom = 12;
    }
    if (now) {
       top_text_bottom += textH;
       sprintf(temp, "Now %s", now_text);
       center_text (x, top_text_bottom, temp);
//       top_text_bottom = 23;
    }
  }

  else { // Nothing fits, use two lines per
    if (iscurrent) {
      center_text (x, top_text_bottom+textH, "Max Flood");
      center_text (x, top_text_bottom+textH*2, next_hta_text);
      center_text (x, bottom-textH, "Max Ebb");
      center_text (x, bottom, next_lta_text);
      top_text_bottom += textH*2;
    }
    else {
      center_text (x, top_text_bottom+textH, "High Tide");
      center_text (x, top_text_bottom+textH*2, next_hta_text);
      center_text (x, bottom-textH, "Low Tide");
      center_text (x, bottom, next_lta_text);
      top_text_bottom += textH*2;
    }
    if (now) {
      center_text (x, top_text_bottom+textH, "Time Now");
      center_text (x, top_text_bottom+textH*2, now_text);
      top_text_bottom += textH*2;
    }
  }
}

/* Draw text telling when the mark transition occurred.  Only called
   by graph mode. */
static void
write_trans_time (int x, time_t tide_time)
{
  int bottom = winH-1;
  if (lines) {
    bottom -= 10;
    if (tinc)
      bottom -= textH;
  }
  if (!onedate || resetdate != get_tide_day(tide_time)) {
//      resetdate = get_tide_day(tide_time);
    if ((event_type & 8) && stagger) // Rising level
      center_text (x, bottom, next_ht_date);
    else
      center_text (x, bottom-textH, next_ht_date);
  }
  if ((event_type & 4) && stagger) // Falling level
    center_text (x, bottom, next_ht_text);
  else
    center_text (x, bottom-textH, next_ht_text);
  return;
}

static void find_tmin_tmax() {
  /* This silly walk is to determine the range within which to draw
     tick marks. */
  if (graphmode && stagger && staggerfloat)
    tmin = textH;
  else
    tmin = textH*2;             //30;
//  if (show_moons)
//    tmin += 10;
  if (caption)
    tmin += textH;
  if (now && !graphmode)
    tmin += textH;//*2;            //24;
  if (graphmode) {
    tmax = winH - textH;        //14;
    if (mark)
      tmax -= textH*2;          //24;
    if (lines) {
      tmax -= 9;
      if (tinc)
        tmax -= textH;          //12;
    }
  }

  else {
    tmax = winH - textH*2;
    if (lines && hairy) {
      tmax -= 9;
      if (tinc)
        tmax -= textH;
    }
//    tmax = winH - 29;
//    if (tinc)
//      tmax -= 12;
//    if (hairy)
//      tmax -= 9;
  }
  tmin = (int) (wl2tide (tmin) * fakeamplitude + fakedatum);
  tmax = (int) (ceil (wl2tide (tmax) * fakeamplitude + fakedatum));
}

/* The unit lines are now calibrated against MLLW and try not to
clobber text.  Jack Greenbaum submitted the original calibrated tick
mark code, which I promptly munged beyond recognition. */
static void
draw_unit_lines ()
{
  int a, b, len, firstflag=1;
  if (!lines)
    return;
  if (graphmode)
    len = winW-1;
  else
    len = 6;

  find_tmin_tmax();

  /* Now get on with it! */
  SelectPen(display,fgtext_color_gc);
  for (a=tmin;a>=tmax;a--) {
    if (fakeamplitude > 30.0 && (a % 10))
      continue;
    b = (int)tide2wl (((double)a - fakedatum) / fakeamplitude);
    if (hinc) {
      if (!(a % hinc)) {
        char temp[20+MAXARGLEN];
        int l;
        DWORD dwExtents;

        make_depth_caption (&firstflag, a, temp);
        l = strlen (temp);
        dwExtents=GetTextExtent(temp,l);
        if (graphmode) {
            MoveTo(display,0,b);
//            LineTo(display,len-(LOWORD(dwExtents)>>1),b);
            LineTo(display,len-LOWORD(dwExtents),b);
            TextOut(display,winW-LOWORD(dwExtents),b-(HIWORD(dwExtents)>>1),temp,l);
        }
        else {
            MoveTo(display,0,b);
            LineTo(display,len,b);
            TextOut(display,len+3,b-(HIWORD(dwExtents)>>1),temp,l);
        }
        continue;
      }
    }
    else
      XDrawLine (display, fgtext_color_gc, 0, b, len, b);
  }
}

/* In graph mode, the numbers labeling the depth lines get clobbered by
   the graph.  This puts them back.  Tmax and tmin are assumed
   to have been set up by draw_unit_lines. */
static void
graph_relabel_unit_lines ()
{
  int a, l, b, firstflag=1;
  char temp[20+MAXARGLEN];
  DWORD dwExtents;

  if (!lines)
    return;
  if (!hinc)
    return;
  for (a=tmin;a>=tmax;a--) {
    if (fakeamplitude > 30.0 && (a % 10))
      continue;
    if (!(a % hinc)) {
      b = (int)tide2wl (((double)a - fakedatum) / fakeamplitude);
      make_depth_caption (&firstflag, a, temp);
      l = strlen (temp);
      dwExtents=GetTextExtent(temp,l);
      TextOut(display,winW-LOWORD(dwExtents),b-(HIWORD(dwExtents)>>1),temp,l);
    }
  }
}

/* Middle and mark and mllw options. */
static void
draw_extra_lines ()
{
  int b;
  if (mark) {
    b = (int)tide2wl ((marklev - fakedatum) / fakeamplitude);
    XDrawLine (display, fgmark_color_gc, 0, b, winW, b);
  }
  if (middle) {
    b = (int)tide2wl (0.0);
    XDrawLine (display, fgmiddle_color_gc, 0, b, winW, b);
  }
  if (mllw) {
    b = (int)tide2wl (-fakedatum / fakeamplitude);
    XDrawLine (display, fgmllw_color_gc, 0, b, winW, b);
  }
  if (iscurrent && !mark) {
    b = (int)tide2wl (-fakedatum / fakeamplitude);
    XDrawLine (display, fgtext_color_gc, 0, b, winW, b);
  }
}

/* Tick marks for hours along X axis */
static void
draw_tick_marks (time_t start)
{
  if (!lines)
    return;
  if ((!graphmode) && (!hairy))
    return;
  if (hairy && !hairycalibrate && !graphmode) {
    /* Center tick marks around current time; ignore start. */
    int a, b;
    char buf[20];
    b = winW>>1;
    XDrawLine (display, fgtext_color_gc, b, winH-7, b, winH-1);
    for (a=1;a<=(int)((winW>>1) * tstep / 3600);a++) {
      b = (winW>>1) + a * 3600 / tstep;
      XDrawLine (display, fgtext_color_gc, b, winH-7, b, winH-1);
      if (tinc) {
        sprintf(buf,"%+d",a);
        center_text (b, winH-10, buf);
      }
      b = (winW>>1) - a * 3600 / tstep;
      XDrawLine (display, fgtext_color_gc, b, winH-7, b, winH-1);
      if (tinc) {
        buf[0] = '-';
        center_text (b, winH-10, buf);
      }
    }
  } else {
    /* Calibrate tick marks with top of the hour */
    int x, hour_mod, ihour, hour_len;
    time_t hour;
    DWORD dwExtents=GetTextExtent("24",2);
    hour_len = HOURSECONDS / tstep;
    if (LOWORD(dwExtents) < hour_len)
         hour_mod = 1;
    else if (LOWORD(dwExtents) < hour_len*2)
         hour_mod = 2;
    else if (LOWORD(dwExtents) < hour_len*4)
         hour_mod = 4;
    else hour_mod = 8;
    for (hour=prev_hour(start);hour<=start+winW*tstep+3600;
         hour=increment_hour(hour)) {
      x = (hour - start) / tstep;
      if (x < winW) {
        XDrawLine (display, fgtext_color_gc, x, winH-7, x, winH-1);
        if (tinc) {
          struct tm *foo;
          foo = tmtime (hour);
          if (1) {// (!((foo->tm_hour) % tinc)) {
            char buf[20];
            do_timestamp (buf, foo);
            if (buf[1] == ':')
                 buf[1] = '\0';
            else buf[2] = '\0';
            ihour = atoi(buf);
            if (!(ihour % hour_mod))
               center_text (x, winH-10, buf);
          }
        }
      }
    }
    /* Make tick marks for day boundaries thicker */
    /* This is not guaranteed to coincide with an hour transition! */
    for (hour=prev_day(start);hour<=start+winW*tstep+3600;
    hour=increment_day(hour)) {
      x = (hour - start) / tstep;
      XDrawLine (display, fgtext_color_gc, x-1, winH-7, x-1, winH-1);
      XDrawLine (display, fgtext_color_gc, x, winH-7, x, winH-1);
      XDrawLine (display, fgtext_color_gc, x+1, winH-7, x+1, winH-1);
    }
  }
}

/*-----------------6/1/2005 5:44AM------------------
 * Find where we are in current tide
 * Returns 0.0 to 1.0 where 0 is lowest tide/current, 1.0 is highest
 * --------------------------------------------------*/
float  where_in_current_tide(time_t this_time) {
double first_tide, now_tide, next_tide;
time_t first_tide_time;

  next_ht = (time_t)(this_time - DAYSECONDS*1.5);
  prev_ht_adj = next_ht_adj = event_type= 0;
  while (next_ht_adj < this_time || (event_type & 3)==0) {
    if ((event_type & 3) != 0)        // Only test for max/min
      first_tide_time = next_ht_adj;
    event_type = update_high_tide ();
  }
  first_tide = time2asecondary (first_tide_time);
  now_tide   = time2asecondary (this_time);
  next_tide  = time2asecondary (next_ht_adj);
  if (first_tide > next_tide)
       return (float)((now_tide - first_tide) / (next_tide  - first_tide));
  else return (float)((now_tide - next_tide ) / (first_tide - next_tide ));
}


/* Handle ugliness relating to high and low tide updates.  This is
   only difficult because (1) I want the timestamps to update at the
   same time that the water changes color, and (2) the change of epoch
   causes special problems. */
static void
watch_tides (time_t this_time)
{
  prev_tide = this_tide;
  this_tide = time2secondary (this_time);
  if (rising == -1) {
    double ttide = time2secondary (this_time+1);
    /* prev_tide is unreliable on startup */
    if (this_tide < ttide)
         rising = 1;
    else if (this_tide == ttide)
         rising = this_tide < time2secondary (this_time+10);
    else rising = 0;
  } else {
    if (prev_tide < this_tide) {
      /* Extra effort to get the tide to update at the same time that the
         water changes color.  The 70 instead of 60 is intentional. */
      if ((!rising) && (this_time > prev_ht_adj - 70)) update_high_tide ();
      rising = 1;
    }
    if (prev_tide > this_tide) {
      /* See above. */
      if (( rising) && (this_time > prev_ht_adj - 70)) update_high_tide ();
      rising = 0;
    }
  }
  /* Clean up any tides that were missed by the above code.  (It happens.) */
  while (this_time >= prev_ht_adj + 70)
    update_high_tide ();
}

/* Redraw the window for the normal tide clock. */
static void
set_water_level ()
{
  int nwl;
  time_t this_time, next_sun_time;
  COLORREF background;
  if (testmode)
       this_time = faketime;
  else this_time = time (NULL);
  window_left_time = this_time;
  next_sun_time = this_time;
  rising = -1;
  background = next_sun_event(&next_sun_time, IDX_lat, IDX_lon, 1)? bgnite_color : bgday_color;
  watch_tides (this_time);
  nwl = (int)tide2wl (this_tide);
  if (iscurrent) {
    int midwl = (int)tide2wl (-fakedatum / fakeamplitude);
    if (nwl < midwl) {
      XFillRectangle (display, background, 0, 0, winW, nwl);
      XFillRectangle (display, background, 0, midwl, winW,winH-midwl);
      XFillRectangle (display, fgrise_color, 0, nwl, winW,midwl-nwl);
    } else if (nwl > midwl) {
      XFillRectangle (display, background, 0, 0, winW, midwl);
      XFillRectangle (display, background, 0, nwl, winW,winH-nwl);
      XFillRectangle (display, fgfall_color,0, midwl+1, winW,nwl-midwl);
    } else {
      XFillRectangle (display, background, 0, 0, winW, winH);
    }
  } else {
    XFillRectangle (display, background, 0, 0, winW, nwl);
    if (rising)
         XFillRectangle (display, fgrise_color, 0, nwl, winW,winH-nwl);
    else XFillRectangle (display, fgfall_color, 0, nwl, winW,winH-nwl);
  }
  write_high_tide (winW>>1, time(NULL));
  draw_unit_lines();
  draw_extra_lines();
  draw_caption();
}

/* Support functions to abstract out the ugly decision tree for */
/* graphing currents. */

static void
x_wl_to_midwl (double wl, int midwl, int looper, int clearflag, HPEN bg_color)
{
  if ((int)wl < midwl) {
    if (!linegraph) {
      if ((int)wl+1 <= midwl-1)
        XDrawLine (display, fgrise_color_gc, looper, (int)wl+1, looper, midwl);
      XDrawLine (display, fgrise_color_gc, looper, (int)wl, looper, (int)wl+1);
    } else
        XDrawLine (display, fgrise_color_gc, looper, (int)wl, looper, midwl-1);
    if (clearflag) {
      XDrawLine (display, bg_color, looper, 0, looper, (int)wl);
      XDrawLine (display, bg_color, looper, midwl+1, looper, winH-1);
    }
  } else if ((int)wl > midwl) {
    if (!linegraph) {
      if ((int)wl >= midwl+1)
        XDrawLine (display, fgfall_color_gc, looper, midwl+1, looper, (int)wl);
      XDrawLine (display, fgfall_color_gc, looper, (int)wl, looper, (int)wl+1);
      if (clearflag) {
        XDrawLine (display, bg_color, looper, 0 , looper, midwl);
        XDrawLine (display, bg_color, looper, (int)wl+1, looper, winH-1);
      }
    } else {
      XDrawLine (display, fgfall_color_gc, looper, midwl+1, looper, (int)wl);
      if (clearflag) {
        XDrawLine (display, bg_color, looper, 0, looper, midwl-1);
        XDrawLine (display, bg_color, looper, (int)wl+1, looper, winH-1);
      }
    }
  } else {
    if (clearflag)
      XDrawLine (display, bg_color, looper, 0, looper, winH-1);
  }
}

static void
x_current_graph (double nwl, double owl, int midwl, int looper, int clearflag, HPEN bg_color)
{
  if (linegraph) {
    if (clearflag)
      XDrawLine (display, bg_color, looper, 0, looper, winH-1);
    if (nwl < midwl) {
      if (owl < midwl)
        XDrawLine (display, fgrise_color_gc, looper-1, (int)owl, looper, (int)nwl);
      else {
        x_wl_to_midwl (owl, midwl, looper-1, 0, bg_color);
        x_wl_to_midwl (nwl, midwl, looper,   0, bg_color);
      }
    } else if (nwl > midwl) {
      if (owl > midwl)
        XDrawLine (display, fgfall_color_gc, looper-1, (int)owl, looper, (int)nwl);
      else {
        x_wl_to_midwl (owl, midwl, looper-1, 0, bg_color);
        x_wl_to_midwl (nwl, midwl, looper,   0, bg_color);
      }
    } else {
      x_wl_to_midwl (owl, midwl, looper-1, 0, bg_color);
    }
  } else {
    x_wl_to_midwl (nwl, midwl, looper, clearflag, bg_color);
  }
}

/* Similar for normal tides */
static void
x_tide_graph (double nwl, double owl, int looper, double prev_g_tide, double
this_g_tide, int clearflag, HPEN bg_color)
{
  int prevx, prevy;
  if (clearflag)
    XDrawLine (display, bg_color, looper, 0, looper, winH);
  if (linegraph) {
    prevx = looper - 1;
    prevy = (int)owl;
  } else {
    prevx = looper;
    prevy = winH - 1;
  }
  if (/*!hairy ||*/ this_g_tide > prev_g_tide) {
    if (!linegraph)
         XDrawLine (display, fgrise_color_gc, prevx, prevy, looper, (int)nwl+1);
    else XDrawLine (display, fgrise_color_gc, prevx, prevy, looper, (int)nwl);
  } else {
    if (!linegraph)
         XDrawLine (display, fgfall_color_gc, prevx, prevy, looper, (int)nwl+1);
    else XDrawLine (display, fgfall_color_gc, prevx, prevy, looper, (int)nwl);
  }
}

/* BOGUS units conversion - Added mgh
 * For stations with knots^2 values, converts a normalized graphics tide
 * level to match the units lines. */
void check_BOGUS(double *norm) {
double off,n,sqr;
  if (have_BOGUS && !have_offsets && convert_BOGUS) {
    off = -fakedatum / fakeamplitude;
    n   = (*norm - off) * amplitude;      // n = original amplitude
    sqr = sqrt(fabs(n)) / fakeamplitude;  // convert to fake units
    if (n >= 0.0)
         *norm = sqr + off;
    else *norm = off - sqr;
  }
  return;
}

/* Draw graph; don't return. */
static void
draw_graph ()
{
  int bottomlines = depthlines && !toplines;
  int a, /*event_type,*/ midwl, sun_is_up, beg, end;
  double prev_g_tide, this_g_tide, nwl, owl;
  time_t start = faketime, this_time = time(NULL), finish, next_sun_time;
  HPEN bg_color;
  COLORREF background;
  /* Initialize the amplitude. */
  happy_new_year (yearoftimet (start));
  midwl = (int)tide2wl (-fakedatum / fakeamplitude);
  faketime = start;
  next_sun_time = faketime;
  next_sun_event(&next_sun_time, IDX_lat, IDX_lon, -1);
  if (bottomlines) {
    beg = end = 0;
    do {
      sun_is_up = next_sun_event(&next_sun_time, IDX_lat, IDX_lon, 1);
      background  = sun_is_up? bgnite_color : bgday_color;
      end = (next_sun_time - faketime) / tstep;
      if (end > winW)
          end = winW;
      XFillRectangle (display, background, beg, 0, end, winH);
      beg = end;
    }
    while (end < winW);
  }

  faketime = start;
  next_sun_time = window_left_time = faketime;
  next_sun_event(&next_sun_time, IDX_lat, IDX_lon, -1);
  if (bottomlines)
    draw_unit_lines();
  else
    find_tmin_tmax();

  prev_g_tide = time2secondary (faketime-1);
  check_BOGUS(&prev_g_tide);
  owl = tide2wl (prev_g_tide);
  for (a=0;a<winW;a++) {
    if (faketime > next_sun_time) {
       sun_is_up = next_sun_event(&next_sun_time, IDX_lat, IDX_lon, 1);
       bg_color  = sun_is_up? bgnite_color_gc : bgday_color_gc;
    }
    this_g_tide = time2secondary (faketime);
    check_BOGUS(&this_g_tide);
    nwl = tide2wl (this_g_tide);
    if (iscurrent)
      x_current_graph (nwl, owl, midwl, a, !bottomlines, bg_color);
    else
      x_tide_graph (nwl, owl, a, prev_g_tide, this_g_tide, !bottomlines, bg_color);
    prev_g_tide = this_g_tide;
    owl = nwl;
    faketime += tstep;
  }
  if (depthlines && toplines)
    draw_unit_lines();
  else
    graph_relabel_unit_lines ();
  draw_extra_lines();
  /* Do this first so red marks will go on top. */
  draw_tick_marks(start);
  if (1) {// Always make an X// (now) {
    /* X marks the spot */
    int x, y;
    x = (int)((this_time - start) / tstep);
    this_g_tide = time2secondary (this_time);
    check_BOGUS(&this_g_tide);
    y = (int)tide2wl (this_g_tide);
    XDrawLine (display, fgtext_color_gc, x, y-4, x, y+4);
    XDrawLine (display, fgtext_color_gc, x-4, y, x+4, y);
  }
  next_ht = start - MAX(abs(httimeoff),abs(lttimeoff));
  finish = start + winW*tstep + MAX(abs(httimeoff),abs(lttimeoff));
  event_type = update_high_tide ();
  resetdate = 0;
  while (next_ht < finish) {
    int x = (int)((next_ht_adj - start) / tstep);
    if (event_type & 3)    /* 1 = low, 2 = high */
      write_high_tide (x, next_ht_adj);
    if (event_type & 12) { /* 4 = falling, 8 = rising */
      write_trans_time (x, next_ht_adj);
      if (lines)
        XDrawLine (display, fgmark_color_gc, x, winH-7, x, winH-1);
    }
    event_type = update_high_tide ();
  }
  draw_moons();
  draw_caption();
}

/* Redraw the window for the hairy tide clock. */
static void
set_hairy_water_level ()
{
  int looper, w2, midwl, sun_is_up;
  time_t this_time, step_time, next_sun_time, finish;
  double prev_g_tide, this_g_tide, nwl, owl;
  HPEN bg_color;
  w2 = winW>>1;
  if (testmode)
       this_time = faketime;
  else this_time = time (NULL);
  next_sun_time = window_left_time = this_time - tstep * (w2+1);
  next_sun_event(&next_sun_time, IDX_lat, IDX_lon,-1);
  rising = -1;
  watch_tides (this_time);
  prev_g_tide = time2secondary (this_time - tstep * (w2+1));
  check_BOGUS(&prev_g_tide);
  owl = tide2wl (prev_g_tide);
  midwl = (int)tide2wl (-fakedatum / fakeamplitude);
  for (looper=0;looper<winW;looper++) {
    step_time = this_time - tstep * (w2-looper);
    this_g_tide = time2secondary (step_time);
    if (step_time > next_sun_time) {
      sun_is_up = next_sun_event(&next_sun_time, IDX_lat, IDX_lon, 1);
      bg_color  = sun_is_up? bgnite_color_gc : bgday_color_gc;
    }
    check_BOGUS(&this_g_tide);
    nwl = tide2wl (this_g_tide);
    if (iscurrent)
      x_current_graph (nwl, owl, midwl, looper, 1, bg_color);
    else
      x_tide_graph (nwl, owl, looper, prev_g_tide, this_g_tide, 1, bg_color);
    prev_g_tide = this_g_tide;
    owl = nwl;
  }
  next_ht = this_time - MAX(abs(httimeoff),abs(lttimeoff));
  do update_high_tide();
  while (next_ht_adj < this_time); // Find first "next" tide
  update_high_tide ();//mgh+ Get next 2 tides from NOW
  write_high_tide (winW>>1, next_ht_adj);
  draw_unit_lines();
  draw_extra_lines();
  draw_tick_marks(this_time - tstep * w2);
  /* X marks the spot */
  this_g_tide = time2secondary (this_time);
  check_BOGUS(&this_g_tide);
  nwl = tide2wl (this_g_tide);
  XDrawLine (display, fgtext_color_gc, w2, (int)nwl-4, w2, (int)nwl+4);
  XDrawLine (display, fgtext_color_gc, w2-4, (int)nwl, w2+4, (int)nwl);

  if (mark && lines) {
    next_ht = window_left_time - MAX(abs(httimeoff),abs(lttimeoff));
    finish = window_left_time + winW*tstep + MAX(abs(httimeoff),abs(lttimeoff));
    graphmode = 1; // Allow mark processing
    event_type = update_high_tide ();
    while (next_ht < finish) {
      if (event_type & 12) {
        int x = (int)((next_ht_adj - window_left_time) / tstep);
        XDrawLine (display, fgmark_color_gc, x, winH-7, x, winH-1);
      }
      event_type = update_high_tide ();
    }
    graphmode = 0;
  }
  draw_moons();
  draw_caption();
}

/*-----------------10/3/2002 6:44PM-----------------
 *
 * --------------------------------------------------*/
void NextEventAdjusted( time_t *ttm, int direction ) {
time_t told, start;
  start = *ttm;
  next_ht = start - MAX(abs(httimeoff),abs(lttimeoff));
  if (direction > 0) {
    do update_high_tide ();
    while (next_ht_adj <= start);
    *ttm = next_ht_adj;
  }
  else {
    next_ht -= DAYSECONDS*2;
    do {
      told = next_ht_adj;
      update_high_tide();
    } while (next_ht_adj < start);
    *ttm = told;
  }
}

/* Redraw the window for the hairy tide clock. */

DWORD WINAPI MyMessageLoop(LPVOID lpv) {
  MSG msg;

  while(GetMessage(&msg,NULL,0,0)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
  }
  return 0;
}

void crash(void) {
  if(szAccum)
      MessageBox(NULL,szAccum,"Tidelib Fatal Error:",MB_ICONSTOP);
}

void list_switches() {
   printf (
"Usage:\n\
wxtide32 [-24]                  Use 24-hour time, not AM / PM\n\
         [-append]              Append - WXTide32 to end of window name\n\
         [-autokt]              Auto change knots^2 to knots\n\
         [-autosave]            Save changed options and window positions on exit\n\
         [-banner]              Sideways ASCII tide graph\n\
         [-bgday ...]           Background color for day time\n\
         [-bgnite ...]          Background color for night time\n\
         [-bmp {filename}       Name of BMP file to generate from graphics window\n\
         [-calen [N]]           Make month tide calendar for N months\n\
         [-calib]               (Hairy)  Tick marks at top of the hour\n\
         [-caption]             (Graph) Displays station name at top of window\n\
         [-check YYYY]          Check for errors in equilibrium arguments\n\
         [-config {filename | -}]  Read configuration file / stdin\n\
         [-cur]                 Restrict -location search to currents\n\
         [-custname \"name\"]     Name of custom station, only used with custom\n\
                                station offsets (-htoff, -hloff, -ltoff, -htoff)\n\
         [-datemdy]             Shows dates as mm-dd-yyyy instead of yyyy-mm-dd\n\
         [-depthlines]          Shows depth lines on graph\n\
         [-fgfall ...]          Color of ebbing water\n\
         [-fgmark ...]          Color of mark line\n\
         [-fgmapdot ...]        Color of tide station dots on locator map\n\
         [-fgmcudot ...]        Color of current station dots on locator map\n\
         [-fgmiddle ...]        Color of middle line\n\
         [-fgmllw ...]          Color of mllw line\n\
         [-fgrise ...]          Color of flooding water\n\
         [-fgtext ...]          Color of text and lines\n");
   printf ("\
         [-graph]               Main and Text Mode is graph tides\n\
         [-gstart YYYY:MM:DD:HH:MM]  Time at which to start graph or text\n\
         [-gstretch N.NN]       Adjust graph detail vs. time span\n\
         [-hairy]               Main Mode is centered tide graph\n\
         [-help]                Send usage to standard output\n\
         [-hfile ...]           Location and name of harmonics file\n\
         [-hinc [N]]            Label depth marks in increments of N\n\
         [-hloff [{-|x}]N.NN]   Tide level offset for high tide\n\
         [-htoff [-]HH:MM]      Time offset for high tide\n\
         [-iconstyle N          Style of tooltray icon 0:line, 1:arrow\n\
         [-incremental [minutes]] Show incremental tide levels\n\
         [-indexfile {filename} Name of index file\n\
         [-list [N]             List all locations in index N1=DMS, N2=DMMmm\n\
         [-lloff [{-|x}]N.NN]   Tide level offset for low tide\n\
         [-location \"Name\"]     Location to show tides for\n\
         [-loctz]               Try to use time zone of location (Moof!)\n\
         [-ltoff [-]HH:MM]      Time offset for low tide\n\
         [-mapcues [N]          Enable map station cues, 0=None, 1=Hover, 2=fast\n\
         [-mapgrid              Enable WVS map grid lines\n\
         [-mapmpy               Station map initial zoom multiplier (0-4)\n\
         [-mark [-]N.NN]        Draw line at specified tide level\n\
         [-marklevel [-]N.NN]   Sets mark level independent of -mark\n\
         [-maxmru N]            Max most recently used entries saved (0-9)\n\
         [-middle]              Draw line near mean tide level\n\
         [-mllw]                Draw line at mean lower low water level\n\
         [-moon]                Show moon phases on graphs\n");
   printf ("\
         [-nearest LAT LON]     Auto-selects nearest station to Lat/Lon (D.MMmm)\n\
         [-nofill]              Line graph instead of filled-in graph\n\
         [-nolines]             Suppress tick marks / depth lines\n\
         [-now]                 Show current time / mark place on graph\n\
         [-nowarn]              Suppress big ugly warning message\n\
         [-numdays [N]]         Number of days of text tides\n\
         [-nummonths [N]]       Number of months of calendar tides\n\
         [-onedate]             Graph mode declutter to show date once\n\
         [-onesc N]             ESCape key action is 0:None,1:Minimize,2:Exit\n\
         [-onlyone]             Only allow one instance of program\n\
         [-onlytcd N]           Only show binary TCD file locations when N=1\n\
         [-overview]            Selects overview graphics mode\n\
         [-ps]                  Generate PostScript[tm] output\n\
         [-raw YYYY:MM:DD:HH:MM [HH:MM]] Raw output from gstart to time, opt step\n");
   printf ("\
         [-scrollbar]           Show time scrollbar in graph mode\n\
         [-skinny]              Trim the fat, make window tall and skinny\n\
         [-showlevels]          Show tide levels on graphs and clocks\n\
         [-stats YYYY:MM:DD:HH:MM]  Find stats from gstart to this time\n\
         [-stagger]             Graph mode declutter, staggers high/low tide times\n\
         [-staggerfloat]        Graph mode declutter, tide times float on tide\n\
         [-subproc N]           Set subordinate proc to 0:Fast, 1:NOAA\n\
         [-sunmoon]             Show sun/moon rise/fall on text tides\n\
         [-test [N]]            Test mode (mostly useless)\n\
         [-text [N]]            List N tides\n\
         [-textascii [N]]       List N tides / select ASCII graph mode\n\
         [-textfile filepath]   Redirect text output to file, -nowarn inhibits msg\n\
         [-thin]                Leave out year and time zone\n\
         [-tinc [N]]            Label each N hours\n\
         [-toooltray N]         Animated icon 0:None,1:on Minimize,2:Always\n\
         [-toplines]            (Graph)  Depth lines on top of graph\n\
         [-tz [-]HH:MM]         Fixed time zone offset for timestamps\n\
         [-units {ft|m}]        Convert units where applicable\n\
         [-userfile {filename}  Name of user station index file\n\
         [-utc]                 Show timestamps in UTC\n\
         [-uutc]                User's times (gstart, etc.) are in UTC\n\
         [-version]             Print WXTide32 version\n\
         [-weekday]             Show day of week in printed dates\n\
\n");
   printf ("\
Time format example -- half past midnight, June 1, 1995:  1995:06:01:00:30\n\
Colors are specified as rgb:hh/hh/hh (where hh is a 2-digit hexadecimal number).\n");
}

/* -----------------2/12/98 7:07AM-------------------
 * File management stuff added for drag and drop.
 * --------------------------------------------------*/

/* get_path_only transfers the path only to a destination */
void get_path_only(char *dst, char *filename) {
int i;
char ch;

  strcpy(dst, filename);
  for (i=strlen(dst)-2; (i>0) && // Strip program name to just leave path
      ((ch=dst[i]) != '\\') && (ch != ':') ; i--) dst[i] = '\0';
}

/* get_name_only transfers the name only to a destination */
void get_name_only(char *dst, char *filename, WORD bufsize) {
WIN32_FIND_DATA ffbuf;
HANDLE hfind;
  if((hfind = FindFirstFile( filename, &ffbuf)) != INVALID_HANDLE_VALUE) {
     strcpy(dst, ffbuf.cFileName);
     FindClose(hfind);
  }
  else
     GetFileTitle( filename, dst, bufsize); // This started faulting ??????
}

int check_file_path(char *dst, char*filename) {
HFILE hf;
OFSTRUCT ofn;
  if (HFILE_ERROR != (hf=OpenFile(filename, &ofn, OF_EXIST))) {
    strcpy(dst, ofn.szPathName);
    if (!strlen(szAlternatePath)) // Only set this once!
       get_path_only(szAlternatePath, ofn.szPathName);//filename);
//    _lclose(hf);
    return(1);
  }
  sprintf(dst, "%s%s", szProgramPath, filename);
  if (HFILE_ERROR != (hf=OpenFile(dst, &ofn, OF_EXIST))) {
    strcpy(dst, ofn.szPathName);
//    _lclose(hf);
    return(1);
  }
  sprintf(dst, "%s%s", szAlternatePath, filename);
  if (HFILE_ERROR != (hf=OpenFile(dst, &ofn, OF_EXIST))) {
    strcpy(dst, ofn.szPathName);
//    _lclose(hf);
    return(1);
  }
/* File not found, just leave it the way it is */
  else
    strcpy(dst, filename);
  return(0);
}

/* -----------------2/12/98 9:33AM-------------------
 * init_OPENFILENAME will initialize an OPENFILENAME structure
 * for file open or save as.
 * --------------------------------------------------*/
void init_OPENFILENAME(int flags, char *title) {
static int i = -10;
static char szFile[256], szFileTitle[256], szFilter[256];

  strcpy(szFilter,"WXTide32 config files (*.wxt;*.cfg)|*.wxt;*.cfg|");
  for ( i=0; szFilter[i] != '\0'; i++)
     if (szFilter[i] == '|') szFilter[i] = '\0';
  if (have_user_offsets && have_new_custom)
     get_name_only( szFile, custom_name, sizeof(szFile) );
  else if (strlen(szConfigPath))
     get_name_only( szFile, szConfigPath, sizeof(szFile) );
  else if (i == -10) szFile[0] = '\0';
  if (!strlen(szAlternatePath)) // Make sure AlternatePath points somewhere
     strcpy(szAlternatePath, szProgramPath);

  memset (&ofn, 0, sizeof(OPENFILENAME));
  ofn.lStructSize    = sizeof(OPENFILENAME);
  ofn.hwndOwner      = hwndMain;
  ofn.lpstrFilter    = szFilter;
  ofn.nFilterIndex    = 1;
  ofn.lpstrFile      = szFile;
  ofn.nMaxFile       = sizeof(szFile);
  ofn.lpstrFileTitle = szFileTitle;
  ofn.nMaxFileTitle  = sizeof(szFileTitle);
  ofn.lpstrTitle     = title;
  ofn.lpstrInitialDir= szAlternatePath;
  ofn.Flags = flags | OFN_NOCHANGEDIR | OFN_HIDEREADONLY;
  ofn.lpstrDefExt = "wxt";
}

/*-----------------12/26/2002 2:39PM----------------
 * Return number of current section
 * --------------------------------------------------*/
int section_what() {
  if      (hairy)     section_number = MODE_HAIRY;
  else if (overview)  section_number = MODE_OVERVIEW;
  else if (graphmode) section_number = MODE_GRAPH;
  else                section_number = MODE_CLOCK;
  return(section_number);
}
/*-----------------12/26/2002 2:03PM----------------
 * Initialize section to stable values
 * --------------------------------------------------*/
void section_init() {
have_sections = 1;
//section[section_num].name           = "???";
  noampm         = 0;
  caption        = 0;
  hairycalibrate = 0;
  depthlines     = 0;
  tstep          = 180;
  hinc           = 0;
  mark           = 0;
  marklev        = 0.0;
  middle         = 0;
  mllw           = 0;
  show_moons     = 0;
  linegraph      = 0;
  lines          = 1;
  now            = 0;
  onedate        = 0;
  usescrollbar   = 0;
  showampl       = 0;
  skinny         = 0;
  stagger        = 0;
  staggerfloat   = 0;
  tinc           = 0;
  toplines       = 0;
  weekday        = 0;
//  utc            = 0;
}

/*-----------------12/26/2002 2:03PM----------------
 * Write local variables into section variables
 * --------------------------------------------------*/
void section_write(int section_num) {
section_entry *pSection = &section[section_num];
  if (section_num < 0 || section_num > MAX_SECTION) return;
  pSection->noampm         = noampm        ;
  pSection->caption        = caption       ;
  pSection->hairycalibrate = hairycalibrate;
  pSection->depthlines     = depthlines    ;
  pSection->tstep          = tstep         ;
  pSection->hinc           = hinc          ;
  pSection->mark           = mark          ;
  pSection->marklev        = marklev       ;
  pSection->middle         = middle        ;
  pSection->mllw           = mllw          ;
  pSection->show_moons     = show_moons    ;
  pSection->linegraph      = linegraph     ;
  pSection->lines          = lines         ;
  pSection->now            = now           ;
  pSection->onedate        = onedate       ;
  pSection->usescrollbar   = usescrollbar  ;
  pSection->showampl       = showampl      ;
  pSection->skinny         = skinny        ;
  pSection->stagger        = stagger       ;
  pSection->staggerfloat   = staggerfloat  ;
  pSection->tinc           = tinc          ;
  pSection->toplines       = toplines      ;
  pSection->weekday        = weekday       ;
//  pSection->utc            = utc           ;
//  if (hold_main_window || (!IsIconic(hwndMain) && !IsZoomed(hwndMain))) {
  if ((hold_main_window || (!IsIconic(hwndMain) && !IsZoomed(hwndMain)) && (WinDimMain.left > 0 || WinDimMain.right > 0))) {
    pSection->WinDimMain.left  = WinDimMain.left  ;
    pSection->WinDimMain.right = WinDimMain.right ;
    pSection->WinDimMain.top   = WinDimMain.top   ;
    pSection->WinDimMain.bottom= WinDimMain.bottom;
  }
}

/*-----------------12/26/2002 2:03PM----------------
 * Read section variables into local variables
 * --------------------------------------------------*/
void section_read(int section_num) {
section_entry *pSection = &section[section_num];
  if (section_num < 0 || section_num > MAX_SECTION) return;
  noampm         = pSection->noampm         ;
  caption        = pSection->caption        ;
  hairycalibrate = pSection->hairycalibrate ;
  depthlines     = pSection->depthlines     ;
  tstep          = pSection->tstep          ;
  hinc           = pSection->hinc           ;
  mark           = pSection->mark           ;
  marklev        = pSection->marklev        ;
  middle         = pSection->middle         ;
  mllw           = pSection->mllw           ;
  show_moons     = pSection->show_moons     ;
  linegraph      = pSection->linegraph      ;
  lines          = pSection->lines          ;
  now            = pSection->now            ;
  onedate        = pSection->onedate        ;
  usescrollbar   = pSection->usescrollbar   ;
  showampl       = pSection->showampl       ;
  skinny         = pSection->skinny         ;
  stagger        = pSection->stagger        ;
  staggerfloat   = pSection->staggerfloat   ;
  tinc           = pSection->tinc           ;
  toplines       = pSection->toplines       ;
  weekday        = pSection->weekday        ;
//  utc            = pSection->utc            ;
  WinDimMain.left  = pSection->WinDimMain.left;
  WinDimMain.right = pSection->WinDimMain.right;
  WinDimMain.top   = pSection->WinDimMain.top;
  WinDimMain.bottom= pSection->WinDimMain.bottom;
}

/*-----------------12/26/2002 3:16PM----------------
 * Update current section variables
 * --------------------------------------------------*/
void section_save() {
  section_write(section_what());
}

/*-----------------12/26/2002 3:16PM----------------
 * Restore mode variables for a given section
 * --------------------------------------------------*/
int section_restore() {
  if (have_sections) {
    section_read(section_what());
    if (hinc) hincmagic = 1;
//    fudge_constituents(youwant);
    amplitude = 0.0;                // Force multiplier re-compute
    happy_new_year (yearoftimet (time(NULL)));//Force new multipliers
    winX = WinDimMain.left;
    winY = WinDimMain.top;
    winW = WinDimMain.right  - WinDimMain.left;
    winH = WinDimMain.bottom - WinDimMain.top;
//    hold_restore = 1;
    if (!IsIconic(hwndMain) && !IsZoomed(hwndMain))
      MoveWindow(hwndMain,winX,winY,winW,winH,TRUE);
    return(1);
  }
  return(0);
}

/* Convert level (ft/m) value to/from current units */
//void convert_level_units(double *level, int tofeet) {
//   if (units && (tolower(units[0]) == 'm')) {
//      if (tofeet) *level *= 0.3048;
//      else        *level /= 0.3048;
//   }
//}

/* -----------------2/12/98 10:32AM------------------
 * Routines to write current configuration switches to a file
 * --------------------------------------------------*/
void write_sw(HFILE hf, char *sw, int nocomment) {
char s[128], comment;
  if (nocomment)
       comment = ' ';
  else comment = '#';
  sprintf(s, "%c-%s\r\n", comment, sw);
  _lwrite(hf,s,strlen(s));
}

void write_sw_int(HFILE hf, char *sw, int i, int nocomment) {
char s[128], comment;
  if (nocomment)
       comment = ' ';
  else comment = '#';
  sprintf(s, "%c-%s %d\r\n", comment, sw, i);
  _lwrite(hf,s,strlen(s));
}

void write_sw_str(HFILE hf, char *sw, char *str, int nocomment) {
char s[128], comment;
  if (nocomment)
       comment = ' ';
  else comment = '#';
  sprintf(s, "%c-%s %s\r\n", comment, sw, str);
  _lwrite(hf,s,strlen(s));
}

void write_sw_quote(HFILE hf, char *sw, char *str) {
char s[128];
  if (strlen(str) > MAXARGLEN) str[MAXARGLEN-1] = '\0'; // Make sure string fits MAXARG
  sprintf(s, " -%s \"%s\"\r\n", sw, str);
  _lwrite(hf,s,strlen(s));
}

void write_sw_rgb(HFILE hf, char *sw, long int color) {
unsigned char s[128], r,g,b;
  r = GetRValue(color);
  g = GetGValue(color);
  b = GetBValue(color);
  sprintf(s, " -%s rgb:%02x%/%02x%/%02x\r\n", sw, r,g,b);
  _lwrite(hf,s,strlen(s));
}

void write_sw_hhmm(HFILE hf, char *sw, long int sec, int nocomment) {
char s[128], sign, comment;
int hr, min;

  if (nocomment)
       comment = ' ';
  else comment = '#';
  if (sec < 0)
       sign = '-';
  else sign = '+';
  hr = abs(sec) / 3600;
  min= (abs(sec)/60) % 60;
  sprintf(s, "%c-%s %c%1d:%02d\r\n", comment, sw, sign, hr, min);
  _lwrite(hf,s,strlen(s));
}

void write_sw_win(HFILE hf, char *sw, RECT win) {
char s[128], comment;
  comment = ' ';
  sprintf(s, "%c-%s %d %d %d %d\r\n",comment,sw,win.left,win.right,win.top,win.bottom);
  _lwrite(hf,s,strlen(s));
}

void write_config_file_common( HFILE hf) {
//double dbl;
char s[256], *p;
int i;
//  dbl = Ihtleveloff;
//  convert_level_units(&dbl, 1);
  sprintf(s, "*%0.2lf %0.2lf", Ihlevelmult, Ihtleveloff);
  write_sw_str(hf, "hloff", s, have_user_offsets);

//  dbl = Iltleveloff;
//  convert_level_units(&dbl, 1);
  sprintf(s, "*%0.2lf %0.2lf", Illevelmult, Iltleveloff);
  write_sw_str(hf, "lloff", s, have_user_offsets);

  write_sw_hhmm(hf,"htoff", Ihttimeoff, have_user_offsets);
  write_sw_hhmm(hf,"ltoff", Ilttimeoff, have_user_offsets);

  if (have_user_offsets) {
    write_sw_quote(hf, "location", IDX_reference_name);
    write_sw_quote(hf, "custname", custom_name);
  }
  else {
    sprintf(s,"%s (%c)",IDX_station_name,IDX_type);
    write_sw_quote(hf, "location", s);
    write_sw(hf, "custname \"name\"", FALSE);
  }
  write_sw    (hf, "autokt",    convert_BOGUS);
  write_sw    (hf, "autosave",  auto_save);
//  write_sw    (hf, "bells",     ShipBells);
  write_sw_rgb(hf, "bgday",     bgday_color);
  write_sw_rgb(hf, "bgnite",    bgnite_color);
  write_sw    (hf, "cur",       curonly);
  write_sw    (hf, "datemdy",   datemdy);
  write_sw_rgb(hf, "fgfall",    fgfall_color);
  write_sw_rgb(hf, "fgmapdot",  fgmapdot_color);
  write_sw_rgb(hf, "fgcurdot",  fgcurdot_color);
  write_sw_rgb(hf, "fgmark",    fgmark_color);
  write_sw_rgb(hf, "fgmiddle",  fgmiddle_color);
  write_sw_rgb(hf, "fgmllw",    fgmllw_color);
  write_sw_rgb(hf, "fgrise",    fgrise_color);
  write_sw_rgb(hf, "fgtext",    fgtext_color);
  write_sw    (hf, "graph",     graphmode);
  write_sw    (hf, "hairy",     hairy);
  write_sw_int(hf, "iconstyle", IconStyle, TRUE);
  write_sw_int(hf, "incrementstep", increment_step, TRUE);

  if (strnicmp(szProgramPath, hfile_name, strlen(szProgramPath)))
       write_sw_quote(hf, "hfile", hfile_name);
  else write_sw_quote(hf, "hfile", hfile_name+strlen(szProgramPath));

  if (strnicmp(szProgramPath, indexfile_name, strlen(szProgramPath)))
       write_sw_quote(hf, "indexfile", indexfile_name);
  else write_sw_quote(hf, "indexfile", indexfile_name+strlen(szProgramPath));

  if (strnicmp(szProgramPath, userfile_name, strlen(szProgramPath)))
       write_sw_quote(hf, "userfile", userfile_name);
  else write_sw_quote(hf, "userfile", userfile_name+strlen(szProgramPath));

//  write_sw    (hf, "keepindex", keep_index);
  write_sw    (hf, "locmap",   had_map);
  write_sw    (hf, "loctz",    loctz);
  write_sw    (hf, "mapcues",  map_cues);
  write_sw    (hf, "mapgrid",  map_grid);
  write_sw_int(hf, "mapmpy",   mapZoom,    TRUE);
  write_sw_int(hf, "numdays",  num_days,   TRUE);
  write_sw_int(hf, "nummonths",num_months, TRUE);
  write_sw_int(hf, "onesc",    OnESCape,   TRUE);
  write_sw_int(hf, "tooltray", UseIcon,    TRUE);
  write_sw    (hf, "nowarn",   TRUE);// nowarn);
  write_sw    (hf, "onlyone",  OnlyOne);
  write_sw_int(hf, "onlytcd",  OnlyTCD,    TRUE);
  write_sw    (hf, "overview", overview);
  write_sw_int(hf, "subproc",  subproc,    TRUE);
  write_sw    (hf, "sunmoon",  sun_moon);
  write_sw    (hf, "utc",      (utc && !Usetadjust && !loctz));
  if (Usetadjust)
     write_sw_str(hf, "tz", tadjust_last, TRUE);
//  else if (isalpha(tadjust_last[0]))
//     write_sw_str(hf, "tzsave", tadjust_last, TRUE);
  write_sw_str(hf, "units",   youwant, (youwant!=NULL));

  sprintf(s,"# Switches that can only be used on a command line\n");
  _lwrite(hf,s,strlen(s));
  write_sw(hf,     "banner",   FALSE);
  write_sw(hf,     "bmp",      FALSE);
  write_sw(hf,     "calen",    FALSE);
  write_sw(hf,     "gstart",   FALSE);
  write_sw(hf,     "incremental [min]", FALSE);
  write_sw(hf,     "list",     FALSE);
  write_sw(hf,     "listtz",   FALSE);
  write_sw(hf,     "ps",       FALSE);
  write_sw(hf,     "raw",      FALSE);
  write_sw(hf,     "stats",    FALSE);
  write_sw(hf,     "test",     FALSE);
  write_sw(hf,     "text",     FALSE);
  write_sw(hf,     "textascii",FALSE);
//  write_sw(hf,     "tz",     FALSE);
  write_sw(hf,     "uutc",     FALSE);

  sprintf(s,"# Switches used only internally to WXTide32.\n");
  _lwrite(hf,s,strlen(s));
  write_sw_win(hf, "WinDimText", WinDimText);
  write_sw_win(hf, "WinDimMap",  WinDimMap);
  sprintf(s, "%d \"%s\"", gFontSize, gFontName);
  write_sw_str(hf, "gfont", s, TRUE);
  sprintf(s, "%d \"%s\"", tFontSize, tFontName);
  write_sw_str(hf, "tfont", s, TRUE);
//  write_sw(hf,     "append", appendWXTide32);
  for (i = 9; i>=0; i--) {
    p = get_mru_entry( i );
    if (p) write_sw_quote(hf, "mru", p+2);
  }
  write_sw_int(hf, "maxmru", max_mru, TRUE);
}

void write_config_file_section(HFILE hf, int section_num) {
char s[256];

  sprintf(s,"\n-SectionStart %d \"%s\"\n", section_num, section[section_num].name);
  _lwrite(hf,s,strlen(s));
  write_sw    (hf, "24",         noampm);
  write_sw    (hf, "calib",      hairycalibrate);
  write_sw    (hf, "caption",    caption);
  write_sw    (hf, "depthlines", depthlines);
  sprintf(s, "%0.2lf", (180.0 / tstep));
  write_sw_str(hf, "gstretch", s, (tstep != 180));
  write_sw    (hf, "hinc",       hinc);

  sprintf(s, "%0.2lf", marklev);
  write_sw_str(hf, "mark",       s, mark);
  write_sw_str(hf, "marklev",    s, TRUE);
  write_sw    (hf, "middle",     middle);
  write_sw    (hf, "mllw",       mllw);
  write_sw    (hf, "moon",       show_moons);
  write_sw    (hf, "nofill",     linegraph);
  write_sw    (hf, "nolines",    !lines);
  write_sw    (hf, "now",        now);
  write_sw    (hf, "onedate",    onedate);
  write_sw    (hf, "scrollbar",  usescrollbar);
  write_sw    (hf, "showlevels", showampl);
  write_sw    (hf, "skinny",     (skinny==2));
  write_sw    (hf, "stagger",    stagger);
  write_sw    (hf, "staggerfloat", staggerfloat);
  write_sw    (hf, "thin",       (skinny==1));
  write_sw    (hf, "tinc",       tinc);
  write_sw    (hf, "toplines",   toplines);
//  write_sw    (hf, "utc", (utc && !Usetadjust && !loctz));
  write_sw    (hf, "weekday",    weekday);
  write_sw_win(hf, "WinDimMain", WinDimMain);
  sprintf(s,"-SectionEnd %d \"%s\"\n", section_num, section[section_num].name);
  _lwrite(hf,s,strlen(s));
}

void write_config_file(char *filename, int notsaveas) {
HFILE hf;
char s[256], fn[256];//, *p;
int i;

  if (have_params && notsaveas) {
    have_params = FALSE;
    get_name_only(fn, filename, sizeof(fn));
    sprintf(s,"Merge command line switches into \"%s\"?\r\n"
              "Selecting No will cancel the save.", fn);
    if (IDYES != MessageBox(hwndMain, s, "Configuration File Update",
                            MB_ICONQUESTION|MB_YESNO))
      return;
  }
  if (!have_user_offsets && had_user_offsets && notsaveas) {
    had_user_offsets = FALSE;
    get_name_only(fn, filename, sizeof(fn));
    sprintf(s,"Overwrite custom station offsets in \"%s\"?\r\n"
              "Selecting No will cancel the save.", fn);
    if (IDYES != MessageBox(hwndMain, s, "Configuration File Update",
                            MB_ICONQUESTION|MB_YESNO))
      return;
  }
  UserStationFuncs(USF_WRITE, custom_name);
  if (HFILE_ERROR == (hf=_lcreat(filename, 0))) {
    cant_open_config_file = TRUE;
    sprintf(s,"Could not open \"%s\" for write.\r\n"
              "File may be write protected. Turn off read-only bit and try again.",filename);
    MessageBox(hwndMain, s,
              "Configuration File Write Error", MB_ICONEXCLAMATION|MB_OK);
  }
  else {
    section_save();
    cant_open_config_file = FALSE;
    have_params = FALSE;
    new_params = FALSE;
    write_config_file_common(hf);
    for (i=0; i<=MAX_SECTION; i++) {
      section_read(i);
      write_config_file_section(hf, i);
    }
    _lclose(hf);
    section_read(section_what());
//    section_restore();
  }
}

// all our command line handling is going to have to change a bit cos windows
// doesn't really work the same way.
// we will try and allow normal handling of command lines but
// .xtiderc will be replaced by 'wintide.ini' in the windows dir
char *arg;

#define getarg if (!(arg = poparg())) barf (WANTMOREARGS)

int getArgInt(int *i) {
  getarg;
  if (sscanf (arg, "%d", i) != 1) barf (BADINTEGER);
  return *i;
}

int getArgIntIf(int *i, int def) {
  if ( dataarg() ) {
    getarg;
    if (sscanf (arg, "%d", i) != 1) barf (BADINTEGER);
  }
  else if (def!=0) *i = def;
  return *i;
}

float getArgDouble(double *f) {
  getarg;
  if (sscanf (arg, "%lf", f) != 1) barf (BADFLOAT);
  return *f;
}

char * getArgStr(char *s) {
  getarg;
  strcpy(s, arg);
  return s;
}

char * getArgStrADupe(char **s) {
  getarg;
  strADupe(s, arg);
  return *s;
}

int getArgWindow(RECT *w, int v) {
  save_windows = TRUE;
  getArgInt (&w->left);
  getArgInt (&w->right);
  getArgInt (&w->top);
  getArgInt (&w->bottom);
  if (v) ValidateWindowDimensions(w);// GRAPHWIDTH, WINHEIGHT);
  return 1;
}

#define getfloat(a) if (sscanf (arg, "%lf", &a) != 1) barf (BADFLOAT)
#define getcolor(a) {getArgStrADupe(&a);}

int FAR PASCAL WinMain(HINSTANCE hinst,HINSTANCE hprev,LPSTR lpCmdLine,int nCmdShow) {
  MSG msg;
  char szFileName[_MAX_PATH], szTempCmdLine[256];
  char szPathName[_MAX_PATH];
  char *pCmdLine;
  int argl=1, rawstep = 0, i;
  time_t stoptime = 0;
  HACCEL hAccel;
  HWND   hwndprev;

  hwndprev = FindWindowEx(NULL, NULL, "WXTide",NULL);
  g_hinst=hinst;
  atexit(crash);
  set_local_tz(); // LCC doesn't seed current timezone info

  /* From config.h */
  win_assert ((int)(strlen (deflocation)) <= MAXARGLEN);
  strcpy (location, deflocation);

  win_assert ((int)(strlen (hfile)) <= MAXARGLEN);
  strcpy (hfile_name, hfile);

  win_assert ((int)(strlen (indexfile)) <= MAXARGLEN);
  strcpy (indexfile_name, indexfile);

  win_assert ((int)(strlen (userfile)) <= MAXARGLEN);
  strcpy (userfile_name, userfile);

  /* Set program path in case we need it */
  GetModuleFileName(g_hinst,szProgramName, sizeof(szFileName));
  get_path_only(szProgramPath, szProgramName);
  szAlternatePath[0] = '\0';

  /* Check environment */
  if ((arg = getenv ("LOCATION"))) {
    win_assert ((int)(strlen (arg)) <= MAXARGLEN);
    strcpy (location, arg);
  }

  if ((arg = getenv ("HFILE"))) {
    win_assert ((int)(strlen (arg)) <= MAXARGLEN);
    strcpy (hfile_name, arg);
  }

  /* See if we were passed a file name */
  szConfigPath[0] = '\0';
// Cure for being passed a garbage command line.....
  pCmdLine = GetCommandLine();
  while (*pCmdLine == ' ') pCmdLine++; //Strip leading spaces
  if (*pCmdLine != '"')
    while (*pCmdLine != ' ' && *pCmdLine) pCmdLine++;  // Find first space
  else {
    do pCmdLine++;
    while (*pCmdLine != '"' && *pCmdLine); // Find ending quote
    if (*pCmdLine == '"') pCmdLine++;
  }
  strcpy(szTempCmdLine, pCmdLine);

  while (szTempCmdLine[0] == ' ')
    strcpy(szTempCmdLine, szTempCmdLine+1); // Strip leading blanks
  for (i=strlen(szTempCmdLine)-1; (i>0) && (szTempCmdLine[i] <= ' '); i--)
    szTempCmdLine[i] = '\0';                // Strip trailing blanks
// If command line starts with quote or slash or period or alpha character
// THEN assume config file name and parse it
  if ((szTempCmdLine[0] == '"') || (szTempCmdLine[0] == '\\') ||
      (szTempCmdLine[0] == '.') || isalpha(szTempCmdLine[0])) {
    if (szTempCmdLine[0] == '"') {
      for (i=1; (i<strlen(szTempCmdLine) && (szTempCmdLine[i]!='"')); i++) {
          szFileName[i-1] = szTempCmdLine[i];
          szFileName[i  ] = '\0';
      }
    }
    else {
      for (i=0; (i<strlen(szTempCmdLine) && (szTempCmdLine[i]!=' ')); i++) {
          szFileName[i  ] = szTempCmdLine[i];
          szFileName[i+1] = '\0';
      }
    }
    if (i < strlen(szTempCmdLine)) i++;     // Skip over terminator
    strcpy(szTempCmdLine, szTempCmdLine+i); // Strip file name from command line
// If config file name is valid and exists, push those arguments on stack
    if (check_file_path(szPathName, szFileName)) { // File of that name exists
      for (i=0; i<4; i++) szFileName[i] = tolower(szPathName[strlen(szPathName)-4+i]);
      if (!strncmp(szFileName, ".wxt", 4) || (!strncmp(szFileName, ".cfg", 4))) {
    // File exists and has an extention of "wxt" or "cfg" so load it.
        if (push_config_file (szPathName, 1)) {
          argl = 0;
          strcpy( szConfigPath, szPathName );
        }
      }
    }
    if (argl) {
      printf("Ignoring improper command line format: [%s]\n",lpCmdLine);
      strcpy(szTempCmdLine,"");
      nowarn = TRUE;
    }
  }

  have_params = (strlen(szTempCmdLine) && strchr(szTempCmdLine,'-'));

  if (have_params) {

//   BAD Kludge
//   I'm going to stuff the command line into a file so it can be read
//   this is coz windows doesn't handle cmd lines properly

    GetTempPath(sizeof szPathName, szPathName);
    GetTempFileName(szPathName,"tide",0,szFileName);
    hf=_lcreat(szFileName,0);
    win_assert(hf!=HFILE_ERROR);
    _lwrite(hf,szTempCmdLine,strlen(szTempCmdLine));
    _lclose(hf);
    push_config_file(szFileName, 0);
    remove(szFileName);
  }

  /* Try to read "WXTide32.cfg */
  if (argl) {
    char tfile[MAXARGLEN+1], tpath[256];
    sprintf (tfile, "%s", winconfigFN);
    if (check_file_path(tpath, tfile)) { // File of that name exists
      if (push_config_file (tpath, 1)) {
        argl = 0;
        strcpy( szConfigPath, tpath );
      }
    }
    else
      strcpy( szConfigPath, tpath );
  }

  /* If failed, try to read system config file */
  while ((arg = poparg())) {
    if (!strcmp (arg, "-config")) {
      getarg;
      if (!(push_config_file (arg, 1))) barf (CONFIGFAIL);
      else strcpy( szConfigPath, szPathName );
    } else if (!strcmp (arg, "-gfont")) {
      getArgInt(&gFontSize);
      getArgStr(gFontName);
    } else if (!strcmp (arg, "-tfont")) {
      getArgInt(&tFontSize);
      getArgStr(tFontName);
    } else if (!strcmp (arg, "-bg")) {
      getarg;
      strADupe(&bgday_color_arg,  arg);
      strADupe(&bgnite_color_arg, arg);
    } else if (!strcmp (arg, "-hfile")) {
      getarg;
      check_file_path(hfile_name, arg);
    } else if (!strcmp (arg, "-gstart")) {
      getarg;
      faketime = parse_time_string (arg);
      gstart_time = faketime;
    } else if (!strcmp (arg, "-stats")) {
      getarg;
      stoptime = parse_time_string (arg);
    } else if (!strcmp (arg, "-raw")) {
      getarg;
      stoptime = parse_time_string (arg);
      if (dataarg ()) {
        getarg;
        rawstep = hhmm2seconds (arg);
      }
      if (!rawstep) rawstep = 3600;
    } else if (!strcmp (arg, "-test")) {
      testmode = 1;
      getArgIntIf(&testspeed,0);
    } else if (!strcmp (arg, "-textascii")) {
      textascii = 1;
      getArgIntIf(&text,-1);
    } else if (!strcmp (arg, "-hinc")) {
      getArgIntIf(&hinc,0);
      if (hinc < 1) {
        hinc = 1;
        hincmagic = 1;
      }
    } else if (!strcmp (arg, "-subproc")) {
      getArgInt (&subproc);
      if (subproc < 0 || subproc > 2) subproc = 1;
    } else if (!strcmp (arg, "-tinc")) {
      getArgIntIf(&tinc,0);
      if (tinc < 1) tinc = 1;
    } else if (!strcmp (arg, "-tz")) {
      getarg;
      if (isalpha(arg[0])) {
        strcpy(tadjust_last, arg);
        strcpy(tadjust_tzname, arg);
        Usetadjust  = 2;
        utc = loctz = 0;
      }
      else {
        strcpy(tadjust_last, arg);
        tz_time2sec( arg, &Itadjust );
        if (Itadjust) {
          if (Itadjust >  DAYSECONDS) Itadjust =  DAYSECONDS;
          if (Itadjust < -DAYSECONDS) Itadjust = -DAYSECONDS;
          tadjust = Itadjust;
          Usetadjust = 1;
          utc        = 1;
          loctz      = 0;
        }
      }
    } else if (!strcmp (arg, "-tzsave")) {
      getarg;
      if (isalpha(arg[0])) {
        strcpy(tadjust_last  , arg);
        strcpy(tadjust_tzname, arg);
      }
      else {
        strcpy(tadjust_last, arg);
        tz_time2sec( arg, &Itadjust );
      }
    } else if (!strcmp (arg, "-graph")) {
      if (have_sections) {
        section_save();
        overview = 0;
        section_read(MODE_GRAPH);
      }
      graphmode = 1; hairy = 0;
    } else if (!strcmp (arg, "-gstretch")) {
      double gstretch;
      getArgDouble (&gstretch);
      if (gstretch <= 0.0) barf (STRETCHTOOSMALL);
      tstep = (int)(180.0 / gstretch);
      if (tstep <= 0) barf (STRETCHTOOBIG);
    } else if (!strcmp (arg, "-hairy")) {
      if (have_sections) {
        section_save();
        section_read(MODE_HAIRY);
      }
      hairy = 1; graphmode = 0; overview = 0;
    } else if (!strcmp (arg, "-overview")) {
      if (have_sections) {
        section_save();
        section_read(MODE_OVERVIEW);
      }
      overview = 1; hairy = 0; graphmode = 1;
    } else if (!strcmp (arg, "-loctz")) {
      loctz = 1;
      utc = Usetadjust = tadjust = 0;
    } else if (!strcmp (arg, "-calen")) {
      calendar = 1;
      getArgIntIf(&num_months,0);
    } else if (!strcmp (arg, "-mark")) {
      mark = 1;
      getArgDouble (&marklev);
    } else if (!strcmp (arg, "-utc")) {
      utc = 1;
      Usetadjust = tadjust = loctz = 0;
    } else if (!strcmp (arg, "-hloff")) {
      Ihlevelmult = 1.0;
      Ihtleveloff = 0.0;
      do {
         getarg;
         if (arg[0] == '*' || arg[0] == 'x') {
            arg[0] = ' ';
            getfloat (Ihlevelmult);
            if (Ihlevelmult <= 0.0)
               barf (BADMULT);
         }
         else
            getfloat (Ihtleveloff);
      }
      while (dataarg());
    } else if (!strcmp (arg, "-lloff")) {
      Illevelmult = 1.0;
      Iltleveloff = 0.0;
      do {
         getarg;
         if (arg[0] == '*' || arg[0] == 'x') {
            arg[0] = ' ';
            getfloat (Illevelmult);
            if (Illevelmult <= 0.0)
               barf (BADMULT);
         }
         else
            getfloat (Iltleveloff);
      }
      while (dataarg());
    } else if (!strcmp (arg, "-htoff")) {
      getarg;
      Ihttimeoff = hhmm2seconds (arg);
    } else if (!strcmp (arg, "-ltoff")) {
      getarg;
      Ilttimeoff = hhmm2seconds (arg);
    } else if (!strcmp (arg, "-version")) {
      if (PATCHLEVEL)
           printf ("XTide %s.%d\n", VERSION, PATCHLEVEL);
      else printf ("XTide %s\n",    VERSION);
      printf("WXTide32 version based on above XTide source\n");
      done=TRUE;
      goto MessageLoop;
    } else if (!strcmp (arg, "-bmp")) {
      getarg;
      if (!stricmp(arg+(strlen(arg)-4),".bmp"))
        strADupe(&bmp_name, arg);
      else {
        char szMessage[256];
        sprintf(szMessage, "%s%s", arg, ".bmp");
        strADupe(&bmp_name, szMessage);
      }
    } else if (!strcmp (arg, "-indexfile")) {
      getarg;
      check_file_path(indexfile_name, arg);
    } else if (!strcmp (arg, "-userfile")) {
      getarg;
      check_file_path(userfile_name, arg);
    } else if (!strcmp (arg, "-nearest")) {
      nearestto = 1;
      getArgDouble (&Ilon);
      getArgDouble (&Ilat);
    } else if (!strcmp (arg, "-mru")) {
      getarg;
      add_mru( arg );
    } else if (!strcmp (arg, "-SectionStart")) {
      getArgInt (&section_number);
      getarg;
      section_init();
    } else if (!strcmp (arg, "-SectionEnd")) {
      getArgInt (&section_number);
      getarg;
      section_write(section_number);
      section_read(section_what());
    }
    else if (!strcmp(arg,"-24"          )) getArgIntIf(&noampm,1);
    else if (!strcmp(arg,"-append"      )) getArgIntIf(&appendWXTide32,1);
    else if (!strcmp(arg,"-autokt"      )) getArgIntIf(&convert_BOGUS,1);
    else if (!strcmp(arg,"-autosave"    )) getArgIntIf(&auto_save,1);
    else if (!strcmp(arg,"-banner"      )) getArgIntIf(&banner,1);
//    else if (!strcmp(arg,"-bells"       )) getArgIntIf(&ShipBells,1);
    else if (!strcmp(arg,"-bgday"       )) getcolor(bgday_color_arg)
    else if (!strcmp(arg,"-bgnite"      )) getcolor(bgnite_color_arg)
    else if (!strcmp(arg,"-calib"       )) getArgIntIf(&hairycalibrate,1);
    else if (!strcmp(arg,"-check"       )) getArgInt  (&checkyear);
    else if (!strcmp(arg,"-caption"     )) getArgIntIf(&caption,1);
    else if (!strcmp(arg,"-csv"         )) getArgIntIf(&csv,1);
    else if (!strcmp(arg,"-cur"         )) getArgIntIf(&curonly,1);
    else if (!strcmp(arg,"-custname"    )) getArgStrADupe(&custom_name);
    else if (!strcmp(arg,"-datemdy"     )) getArgIntIf(&datemdy,1);
    else if (!strcmp(arg,"-depthlines"  )) getArgIntIf(&depthlines,1);
    else if (!strcmp(arg,"-fgfall"      )) getcolor(fgfall_color_arg)
    else if (!strcmp(arg,"-fgmapdot"    )) getcolor(fgmapdot_color_arg)
    else if (!strcmp(arg,"-fgcurdot"    )) getcolor(fgcurdot_color_arg)
    else if (!strcmp(arg,"-fgmark"      )) getcolor(fgmark_color_arg)
    else if (!strcmp(arg,"-fgmiddle"    )) getcolor(fgmiddle_color_arg)
    else if (!strcmp(arg,"-fgmllw"      )) getcolor(fgmllw_color_arg)
    else if (!strcmp(arg,"-fgrise"      )) getcolor(fgrise_color_arg)
    else if (!strcmp(arg,"-fgtext"      )) getcolor(fgtext_color_arg)
    else if (!strcmp(arg,"-iconstyle"   )) getArgInt  (&IconStyle);
    else if (!strcmp(arg,"-incremental" )) getArgIntIf(&incremental_tides,-1);
    else if (!strcmp(arg,"-incrementstep")) getArgInt (&increment_step);
    else if (!strcmp(arg,"-keepindex"   )) ;//    keep_index = 1;
    else if (!strcmp(arg,"-list"        )) getArgIntIf(&list,1);
    else if (!strcmp(arg,"-listtz"      )) list = 1;
    else if (!strcmp(arg,"-location"    )) getArgStr  (location);
    else if (!strcmp(arg,"-locmap"      )) getArgIntIf(&had_map,1);
    else if (!strcmp(arg,"-mapcues"     )) getArgIntIf(&map_cues,1);
    else if (!strcmp(arg,"-mapgrid"     )) getArgIntIf(&map_grid,1);
    else if (!strcmp(arg,"-mapmpy"      )) getArgIntIf(&mapZoom,0);
    else if (!strcmp(arg,"-marklev"     )) getArgDouble(&marklev);
    else if (!strcmp(arg,"-middle"      )) getArgIntIf(&middle,1);
    else if (!strcmp(arg,"-mllw"        )) getArgIntIf(&mllw,1);
    else if (!strcmp(arg,"-moon"        )) getArgIntIf(&show_moons,1);
    else if (!strcmp(arg,"-nofill"      )) getArgIntIf(&linegraph,1);
    else if (!strcmp(arg,"-nolines"     )) lines = 0;
    else if (!strcmp(arg,"-maxmru"      )) getArgInt  (&max_mru);
    else if (!strcmp(arg,"-now"         )) getArgIntIf(&now,1);
    else if (!strcmp(arg,"-nowarn"      )) getArgIntIf(&nowarn,1);
    else if (!strcmp(arg,"-numdays"     )) getArgInt  (&num_days);
    else if (!strcmp(arg,"-nummonths"   )) getArgInt  (&num_months);
    else if (!strcmp(arg,"-onlyone"     )) getArgIntIf(&OnlyOne,1);
    else if (!strcmp(arg,"-onlytcd"     )) getArgIntIf(&OnlyTCD,1);
    else if (!strcmp(arg,"-onedate"     )) getArgIntIf(&onedate,1);
    else if (!strcmp(arg,"-onesc"       )) getArgInt  (&OnESCape);
    else if (!strcmp(arg,"-ps"          )) getArgIntIf(&ps,1);
    else if (!strcmp(arg,"-scrollbar"   )) getArgIntIf(&usescrollbar,1);
    else if (!strcmp(arg,"-showlevels"  )) getArgIntIf(&showampl,1);
    else if (!strcmp(arg,"-skinny"      )) skinny = 2;
    else if (!strcmp(arg,"-stagger"     )) getArgIntIf(&stagger,1);
    else if (!strcmp(arg,"-staggerfloat")) getArgIntIf(&staggerfloat,1);
    else if (!strcmp(arg,"-sunmoon"     )) getArgIntIf(&sun_moon,1);
    else if (!strcmp(arg,"-text"        )) getArgIntIf(&text,-1);
    else if (!strcmp(arg,"-textfile"    )) getArgStrADupe(&output_filename);
    else if (!strcmp(arg,"-thin"        )) skinny = 1;
    else if (!strcmp(arg,"-tooltray"    )) getArgInt  (&UseIcon);
    else if (!strcmp(arg,"-toplines"    )) getArgIntIf(&toplines,1);
    else if (!strcmp(arg,"-units"       )) getArgStrADupe(&youwant);
    else if (!strcmp(arg,"-uutc"        )) getArgIntIf(&uutc,1);
    else if (!strcmp(arg,"-weekday"     )) getArgIntIf(&weekday,1);
    else if (!strcmp(arg,"-WinDimMain"  )) getArgWindow(&WinDimMain, 1);
    else if (!strcmp(arg,"-WinDimMap"   )) getArgWindow(&WinDimMap,  0);
    else if (!strcmp(arg,"-WinDimText"  )) getArgWindow(&WinDimText, 0);
    else {
      list_switches();
      if (strcmp (arg, "-help"))
         printf("\nERROR: I do not know what switch [%s] means.\n",arg);
      done=TRUE;
      goto MessageLoop;
    }
  }

  /* If onlyone instance option and we were already running then quit */
  if (OnlyOne && hwndprev) {
    if (IsIconic(hwndprev))
         SendMessage (hwndprev, WM_COMMAND, ID_SHOW_HIDE, 0);
    else SetForegroundWindow(hwndprev);
    return 0;
  }

  if (!nowarn) printf (
" WXTide32 (C) 1998-2003 Michael Hopper. Based on XTide (C) 1996 David Flater.\n\
\n\
This program is distributed in the hope that it will be useful, but WITHOUT ANY\n\
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A\n\
PARTICULAR PURPOSE. See the GNU General Public License for more details.\n\
\n\
 *** DO NOT RELY ON THE PREDICTIONS OF THIS PROGRAM FOR DECISIONS THAT CAN ***\n\
 ***      RESULT IN HARM TO ANYONE OR ANYTHING.  REALLY.  I MEAN IT.       ***\n\
  (This warning will disable automatically the first time settings are saved)\n");

  /* Remove glaring inconsistencies in args */
  if (have_sections && !have_params) /* Get data for this type of data view */
    section_read(section_what());

  if (!lines)             hinc = tinc = 0;
  if (banner || stoptime) graphmode = calendar = ps = 0;
  if (graphmode && ps)    graphmode = 0;

//  if (graphmode || ps) {
//    calendar = 0;
//    if (!hinc)
//      hincmagic = 1;
//  }

  if (calendar) {
    incremental_tides = 0;
    text    = 1;
    skinny  = 2;
    weekday = 0;   /* Important!  Don't overrun buffer */
  }

  if (incremental_tides) {
    text    = 1;
    skinny  = 0;
    weekday = 0;   /* Important!  Don't overrun buffer */
  }

  if ( graphmode && text)          skinny = 2;
  if (!graphmode && !hairy && !ps) tinc   = 0;
  if ( hairy && tinc)              hairycalibrate = 1;
  if ( graphmode && text)          tstep *= AGTSTEP;  /* Adjust tstep for ASCII graph mode */
  if ( banner)                     tstep *= BANTSTEP; /* Adjust tstep for banner mode */

  /* I can't do this yet for X graph mode because I don't know the geometry (oops).  See below. */
  if (!(graphmode && !text)) {
    if (!faketime) {                    /* We have NOT been started with -gstart */
      faketime = time (NULL);
      if (graphmode && now)             /* Center graph around current time if -now */
        faketime -= tstep * (TEXTWIDTH>>1);  /* Must be text */
    }
  }

// We leave XTide processing for a moment to do WXTide32 stuff

  ShipBells = haveSounds = check_file_path(HelpFileName, bellsname);
  done=FALSE;
  if (!have_index && !list) {
    init_index_file(keep_index, hwndMain); // Load all or part based in keep_index flag
    if (nearestto) {
      int nearest = load_nearest_station(Ilat, Ilon, 5000);  /* Nearest within 5000 miles */
      if (nearest < 5000)
        new_params = TRUE;
      if (nearest > 10) {
        char s[80];
        sprintf(s, "Closest station is %d NM from -nearest Lon/Lat.\r\n", nearest);
        if (output_filename && nowarn) {
          printf("Error: %s\r\n", s);
          done = TRUE;
        }
        else MessageBox(hwndMain,s,"No near station",MB_ICONINFORMATION);
      }
    }
    else if (!have_index || !load_location_data( location, 0 )) {
      if (output_filename && nowarn) {
         printf("Error: Station \"%s\" is not listed.\r\n", location);
         PostQuitMessage(0);
         done = TRUE;
      }
      else {
         if (!have_index) {
//            MessageBox(hwndMain,"Please select your station from the following menu",
//              "No index file",MB_ICONINFORMATION);
         }
         else {
            char s[256];
            sprintf(s,"Station \"%s\" is not listed.\r\n"
                "Please select another station from the following menu",location);
            MessageBox(hwndMain,s,"Unknown location",MB_ICONINFORMATION);
         }
         remove_mru(location);
         if (!DialogBox(g_hinst,"EnterLocation",hwndMain,LocationDlg))
            PostQuitMessage(0);
      }
    }
  }
//  else load_data ();
  keep_index = 1; // Any stations loaded after the first will load/keep index */

  if(list) {
    list_stations();
    done=TRUE;
    goto MessageLoop;
  }

  if (checkyear) {
    check_epoch ();
    done=TRUE;
    goto MessageLoop;
  }

  /* Warn about the interpolation */
  if (have_offsets && (graphmode || banner || stoptime || !text) && !nowarn)
    printf ("XTide WARNING:  Interpolated tide levels may be inaccurate.\n");

  if (stoptime) {
    if (rawstep)
         do_raw   (faketime, stoptime, rawstep);
    else do_stats (faketime, stoptime);
    done = TRUE;
    goto MessageLoop;
  }

  if (text) {
    if (iscurrent) {
      mark     = 1;
      marklev  = 0.0;
    }
    graphmode= 0;
    if      (textascii)         tide2ascii ();
    else if (calendar)          do_calendar ();
    else if (incremental_tides) do_incremental (csv);
    else if (banner)            do_banner ();
    else                        list_tides (csv);
    done = TRUE;
    goto MessageLoop;
  }

  if (ps) {
    int hours = tstep * 24 / 180;
    if (hours <= 0) barf (STRETCHTOOBIG);
    tide2ps (hours);
    done = TRUE;
    goto MessageLoop;
  }

  bgday_color      =bgday_color_arg?GetColor(bgday_color_arg):RGB(0,255,255);
  bgday_color_gc   =CreatePen(PS_SOLID,0,bgday_color);
  bgnite_color     =bgnite_color_arg?GetColor(bgnite_color_arg):RGB(0,255,255);
  bgnite_color_gc  =CreatePen(PS_SOLID,0,bgnite_color);
  fgrise_color     =fgrise_color_arg?GetColor(fgrise_color_arg):RGB(0,0,255);
  fgrise_color_gc  =CreatePen(PS_SOLID,0,fgrise_color);
  fgfall_color     =fgfall_color_arg?GetColor(fgfall_color_arg):RGB(0,128,0);
  fgfall_color_gc  =CreatePen(PS_SOLID,0,fgfall_color);
  fgtext_color     =fgtext_color_arg?GetColor(fgtext_color_arg):RGB(0,0,0);
  fgtext_color_gc  =CreatePen(PS_SOLID,0,fgtext_color);
  fgmark_color     =fgmark_color_arg?GetColor(fgmark_color_arg):RGB(255,0,0);
  fgmark_color_gc  =CreatePen(PS_SOLID,0,fgmark_color);
  fgmllw_color     =fgmllw_color_arg?GetColor(fgmllw_color_arg):RGB(255,255,255);
  fgmllw_color_gc  =CreatePen(PS_SOLID,0,fgmllw_color);
  fgmiddle_color   =fgmiddle_color_arg?GetColor(fgmiddle_color_arg):RGB(255,255,0);
  fgmiddle_color_gc=CreatePen(PS_SOLID,0,fgmiddle_color);
  fgmapdot_color   =fgmapdot_color_arg?GetColor(fgmapdot_color_arg):RGB(255,0,0);
  fgmapdot_color_gc=CreatePen(PS_SOLID,0,fgmapdot_color);
  fgcurdot_color   =fgcurdot_color_arg?GetColor(fgcurdot_color_arg):RGB(0,0,255);
  fgcurdot_color_gc=CreatePen(PS_SOLID,0,fgcurdot_color);

  if(!hprev) Init_Application(hinst);

  Create_Window (hinst);

  /* Set things up for displaying graphics modes */
  if (graphmode) {
    time_t this_time = time(NULL);
    win_assert (this_time > DAYSECONDS*1.5);
    next_ht     = (time_t)(this_time - DAYSECONDS*1.5);
    prev_ht_adj = next_ht_adj = 0;
    /* See above. */
    if (!faketime) {
      faketime = time (NULL);
      if (now)
        faketime -= tstep * (winW>>1);
    }
//    draw_graph ();
  }
  else {
  /* This clumsy calibration step is necessary for arbitrary offsets in clock mode. */
    time_t this_time = (testmode)? faketime : time(NULL);
    win_assert (this_time > DAYSECONDS*1.5);
    next_ht     = (time_t)(this_time - DAYSECONDS*1.5);
    prev_ht_adj = next_ht_adj = 0;
    while (this_time >= prev_ht_adj + 70)
      update_high_tide ();
    /* Need to do it twice, to get high AND low tides set up */
//    update_high_tide ();
//    update_high_tide ();

  }
  ShowWindow(hwndMain,nCmdShow);
  UpdateWindow(hwndMain);
  hold_main_window = FALSE;
  had_user_offsets = have_user_offsets;
  new_params       = FALSE;
  check_file_path(HelpFileName, "WXTide32.hlp");

  if (!have_sections && !output_filename) { /* If not unattended operation */
    int i;
    new_params = TRUE;
    MessageBox(hwndMain,"This is an older configuration file.\r\n"
              "When you exit, your settings will be saved in the new format.",
              "Configuration conversion",MB_ICONINFORMATION);
    for ( i=0; i<=MAX_SECTION; i++ ) {
      section[i].WinDimMain.left  = WinDimMain.left;
      section[i].WinDimMain.right = WinDimMain.right+i; // Fake out MoveWindow logic
      section[i].WinDimMain.top   = WinDimMain.top;
      section[i].WinDimMain.bottom= WinDimMain.bottom;
    }
    have_sections = 1;
    section_restore();
  }

  if (bmp_name) {
    SaveBitmap(bmp_name, CaptureWindow(hwndMain, winW, winH));
    if (!nowarn) MessageBox(hwndMain,bmp_name,"WXTide32 BMP Saved",MB_ICONINFORMATION);
    done = TRUE;
  }

MessageLoop:
  if (done == TRUE) {
    DWORD dwThreadId;
    HANDLE hThread;

    hThread=CreateThread(NULL,0,MyMessageLoop,0,0,&dwThreadId);
    SetThreadPriority(hThread,THREAD_PRIORITY_ABOVE_NORMAL);
    if (textLFN) fclose( textLFN );
    if (hwndText) MessageBox(hwndMain,"WXTide32 Close","Press any key to exit",MB_ICONINFORMATION);
    return(0);
  }

  hAccel=LoadAccelerators(g_hinst,"WXTIDEACC");
  while(GetMessage(&msg,NULL,0,0)) {
    if(hwndMain!=msg.hwnd || !TranslateAccelerator(hwndMain,hAccel,&msg)) {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
  }
  if (textLFN) fclose( textLFN );
  die();
  return msg.wParam;
}


/*-----------------12/18/2005 11:46AM---------------
 * Update menu items and append the MRU stations
 * --------------------------------------------------*/
void WinTideCheckMenus(HWND hwnd) {
  char *p;
  int   mru, highmru;
  HMENU hmenu=GetMenu(hwnd);

  if(hmenu) {
    CheckMenuItem(hmenu,ID_CLOCK,   MF_BYCOMMAND| ((!graphmode && !hairy)?MF_CHECKED:MF_UNCHECKED));
    CheckMenuItem(hmenu,ID_HAIRY,   MF_BYCOMMAND|                ((hairy)?MF_CHECKED:MF_UNCHECKED));
    CheckMenuItem(hmenu,ID_GRAPH,   MF_BYCOMMAND| ((graphmode&&!overview)?MF_CHECKED:MF_UNCHECKED));
    CheckMenuItem(hmenu,ID_OVERVIEW,MF_BYCOMMAND|             ((overview)?MF_CHECKED:MF_UNCHECKED));
    EnableMenuItem(hmenu,ID_PRINT,  MF_BYCOMMAND|(hwndText!=NULL)?MF_ENABLED:MF_GRAYED);
    EnableMenuItem(hmenu,ID_SAVE,   MF_BYCOMMAND|      new_params?MF_ENABLED:MF_GRAYED);
    EnableMenuItem(hmenu,ID_CUSTOM, MF_BYCOMMAND|      have_index?MF_ENABLED:MF_GRAYED);

    for (highmru=9; highmru>0 && !get_mru_entry(highmru); highmru--)
       DeleteMenu(hmenu, ID_MRU0+highmru, MF_BYCOMMAND);

    for (mru=1; mru<=highmru; mru++) {
      p=get_mru_entry(mru);
      if (!ModifyMenu(hmenu, ID_MRU0+mru, MF_BYCOMMAND|MF_STRING, ID_MRU0+mru, p)) {
           InsertMenu(hmenu, ID_EXIT,     MF_BYCOMMAND|MF_STRING, ID_MRU0+mru, p);
      }
    }
    if (haveSounds == 1) {
      InsertMenu(hmenu, ID_SAVE, MF_BYCOMMAND|MF_STRING, ID_BELLS, "Play ship &bells");
      haveSounds++;
    }
    if (haveSounds) CheckMenuItem(hmenu,ID_BELLS,MF_BYCOMMAND| ((ShipBells)?MF_CHECKED:MF_UNCHECKED));
    DrawMenuBar(hwnd);
  }
}

/*-----------------12/18/2005 11:47AM---------------
 * Preferences helper routines
 * --------------------------------------------------*/
int TZComboIdx;

#define FSAME(a,b) (fabs(a-b) < 0.0001)

int GetDlgItemFloat(HWND hwnd, int ctlID, float *val) {
  char  tmpStr[20];
  GetDlgItemText(hwnd, ctlID, tmpStr, sizeof(tmpStr)-1);
  return (sscanf(tmpStr, "%f", val) == 1);
}

int GetDlgItemDouble(HWND hwnd, int ctlID, double *val) {
  char  tmpStr[20];
  GetDlgItemText(hwnd, ctlID, tmpStr, sizeof(tmpStr)-1);
  return (sscanf(tmpStr, "%lf", val) == 1);
}

char *makeTimeOffset(long sec) {
int hr, min;
static char str[20], sign;

  hr =  abs(sec) / 3600;
  min= (abs(sec) / 60) % 60;
  if (sec < 0)
       sign = '-';
  else sign = '+';
  sprintf(str, "%c%d:%02d", sign, hr, min);
  return(str);
}

char *makeRealStr(double n) {
static char str[20];

  sprintf(str, "%0.2lf", n);
  return(str);
}

/* Turn a time displacement of the form [-]HH:MM into the number of seconds. */
int verify_hhmm2sec(long *dst, char *hhmm) {
int h, m;
char s;
  if (sscanf (hhmm, "%d:%d", &h, &m) != 2) return(1);
  if (sscanf (hhmm, "%c", &s)        != 1) return(1);
  if (h < 0 || s == '-')
    m = -m;
  m %= 60;
  *dst = h*HOURSECONDS + m*60;
  return(0);
}

void LoadTZCombo(HWND hwnd, int ComboBox) {
  char tzname[256];
  int i;

  SendDlgItemMessage(hwnd,ComboBox, CB_RESETCONTENT, 0,0L);
  if (Itadjust) {
    sprintf( tzname, "%s", seconds2hhmm( Itadjust ));
    SendDlgItemMessage(hwnd,ComboBox,CB_INSERTSTRING,-1,(LPARAM)((LPSTR)tzname));
  }
  else if (isalpha(tadjust_last[0])) {
    strcpy( tzname, tadjust_last );
    SendDlgItemMessage(hwnd,ComboBox,CB_INSERTSTRING,-1,(LPARAM)((LPSTR)tzname));
  }
  for (i=0; tz_names[i][0] != NULL; i++)
    SendDlgItemMessage(hwnd,ComboBox,CB_INSERTSTRING,-1,(LPARAM)((LPSTR)tz_names[i][0]));
  if ((Itadjust && Usetadjust != 2) || Usetadjust == 2)
    TZComboIdx = 0;
  else if (strlen(IDX_tzname) > 0)
    TZComboIdx=SendDlgItemMessage(hwnd,ComboBox,CB_FINDSTRING,-1,(LPARAM)((LPSTR)(IDX_tzname)));
  else if (isalpha(tadjust_last[0])) {
    SendDlgItemMessage(hwnd,ComboBox,CB_INSERTSTRING,-1,(LPARAM)((LPSTR)tadjust_last));
    TZComboIdx=SendDlgItemMessage(hwnd,ComboBox,CB_FINDSTRING,-1,(LPARAM)((LPSTR)tadjust_last));
  }
  else
    TZComboIdx=SendDlgItemMessage(hwnd,ComboBox,CB_FINDSTRING,-1,(LPARAM)((LPSTR)(tzfile+1)));
  if (TZComboIdx==CB_ERR) TZComboIdx=0;

  SendDlgItemMessage(hwnd,ComboBox, CB_SETCURSEL, TZComboIdx, 0L);
}

int GetTZCombo(HWND hwnd, int ComboBox, int setter, int onlycheck) {
  int Changed, temptime;//, tzidx;
  char tzname[256];
  GetWindowText(GetDlgItem(hwnd, ComboBox), (LPSTR)&tzname, sizeof(tzname));
  Changed=( TZComboIdx != SendDlgItemMessage(hwnd,ComboBox,CB_FINDSTRING,-1,(LPARAM)((LPSTR)&tzname)) );
  if (onlycheck) return(Changed);
  if (isalpha(tzname[0])) {
    strcpy(tadjust_last, tzname);
    strcpy(tadjust_tzname, tadjust_last);
    if (IsDlgButtonChecked(hwnd,setter))
         Usetadjust = 2;
    else Usetadjust = 0;
    tadjust = 0;
  }
  else {
    tz_time2sec( tzname, &temptime );
    Itadjust = temptime;
    strcpy(tadjust_last, tzname);
    strcpy(tadjust_tzname, tadjust_last);
    if (IsDlgButtonChecked(hwnd,setter)) {
      Usetadjust = 1;
      tadjust    = Itadjust;
    }
    else {
      Usetadjust = 0;
      tadjust    = 0;
    }
  }
  return( Changed );
}

int get_flag(HWND hwnd, int *flag, int dlgitem) {
int changed = 0;
  if (*flag!=IsDlgButtonChecked(hwnd,dlgitem)) {
     *flag = !*flag;
     new_params = TRUE;
     changed    = TRUE;
  }
  return(changed);
}

int get_3st(HWND hwnd, int *flag, int dlgitem) {
int changed = 0;
int flag3 = IsDlgButtonChecked(hwnd,dlgitem) == BST_CHECKED? 1 : 0;
  if ((*flag && !flag3) || (!*flag && flag3)) {
     *flag = !*flag;
     new_params = TRUE;
     changed = TRUE;
  }
  return(changed);
}

void ResetPen(HPEN FAR *lphpen,COLORREF *clr, HWND hwnd, int dlgID) {
  char szBuffer[10];
  GetDlgItemText(hwnd,dlgID,szBuffer,sizeof szBuffer);
  if (*clr != atol(szBuffer)) new_params = TRUE;
  *clr=atol(szBuffer);
  DeletePen(*lphpen);
  *lphpen=CreatePen(PS_SOLID,0,*clr);
}

/*-----------------12/18/2005 11:47AM---------------
 * Preferences Sheet Colors and Fonts page
 * --------------------------------------------------*/
BOOL CALLBACK ColorDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam) {
  char szBuffer[10];
  COLORREF clr;
  static COLORREF aclrCust[16];
  CHOOSECOLOR cc;
  CHOOSEFONT  cf;
  LPDRAWITEMSTRUCT lpdis;
  HBRUSH hbr;
  RECT   rc;
  int    ptSize, i;
  static LOGFONT glf, tlf;
  HDC    hDcDisplay = GetDC(hwnd);
  LPNMHDR lpnmhdr;

  switch(msg) {
    case WM_INITDIALOG:
      for(i=0;i<16;i++)
          aclrCust[i] = RGB(255, 255, 255);

      memset(&glf, 0, sizeof(glf));
      glf.lfHeight = -MulDiv(gFontSize, GetDeviceCaps(hDcDisplay, LOGPIXELSY), 72);
      strcpy(glf.lfFaceName, gFontName);

      memset(&tlf, 0, sizeof(tlf));
      tlf.lfHeight = -MulDiv(tFontSize, GetDeviceCaps(hDcDisplay, LOGPIXELSY), 72);
      strcpy(tlf.lfFaceName, tFontName);

      wsprintf(szBuffer,"%lu",bgday_color);
      SetDlgItemText(hwnd,IDC_BGDAY,szBuffer);
      wsprintf(szBuffer,"%lu",bgnite_color);
      SetDlgItemText(hwnd,IDC_BGNITE,szBuffer);
      wsprintf(szBuffer,"%lu",fgtext_color);
      SetDlgItemText(hwnd,IDC_FGTEXT,szBuffer);
      wsprintf(szBuffer,"%lu",fgrise_color);
      SetDlgItemText(hwnd,IDC_FGRISE,szBuffer);
      wsprintf(szBuffer,"%lu",fgfall_color);
      SetDlgItemText(hwnd,IDC_FGFALL,szBuffer);
      wsprintf(szBuffer,"%lu",fgmiddle_color);
      SetDlgItemText(hwnd,IDC_FGMIDDLE,szBuffer);
      wsprintf(szBuffer,"%lu",fgmark_color);
      SetDlgItemText(hwnd,IDC_FGMARK,szBuffer);
      wsprintf(szBuffer,"%lu",fgmllw_color);
      SetDlgItemText(hwnd,IDC_FGMLLW,szBuffer);
      wsprintf(szBuffer,"%lu",fgmapdot_color);
      SetDlgItemText(hwnd,IDC_FGTDOT,szBuffer);
      wsprintf(szBuffer,"%lu",fgcurdot_color);
      SetDlgItemText(hwnd,IDC_FGCDOT,szBuffer);
      PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, IDOK), -1);
      break;

    case WM_DRAWITEM:
      lpdis=(LPDRAWITEMSTRUCT)lParam;
      GetDlgItemText(hwnd,lpdis->CtlID,szBuffer,sizeof szBuffer);
      clr = atol(szBuffer);
      rc  = lpdis->rcItem;
      hbr = CreateSolidBrush(clr);
      hbr = SelectBrush(lpdis->hDC,hbr);
      Rectangle(lpdis->hDC,rc.left,rc.top,rc.right,rc.bottom);
      DeleteBrush(SelectBrush(lpdis->hDC,hbr));
      if(lpdis->itemState&ODS_FOCUS) {
        InflateRect(&rc,-3,-3);
        DrawFocusRect(lpdis->hDC,&rc);
      }
      break;

    case WM_COMMAND:
      switch(wParam) {

        case IDC_BGDAY:
        case IDC_BGNITE:
        case IDC_FGTEXT:
        case IDC_FGRISE:
        case IDC_FGFALL:
        case IDC_FGMIDDLE:
        case IDC_FGMARK:
        case IDC_FGMLLW:
        case IDC_FGTDOT:
        case IDC_FGCDOT:
          GetDlgItemText(hwnd,wParam,szBuffer,sizeof szBuffer);
          clr = atol(szBuffer);
          memset(&cc, 0, sizeof cc);
          /* Initialize the necessary CHOOSECOLOR members. */
          cc.lStructSize = sizeof cc;
          cc.hwndOwner   = hwnd;
          cc.rgbResult   = clr;
          cc.lpCustColors= aclrCust;
          cc.Flags       = CC_RGBINIT;
          if(ChooseColor(&cc)) {
            wsprintf(szBuffer,"%lu",cc.rgbResult);
            SetDlgItemText(hwnd,wParam,szBuffer);
          }
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDC_GRAF_FONT:
          memset(&cf, 0, sizeof(CHOOSEFONT));
          cf.lStructSize= sizeof(CHOOSEFONT);
          cf.hwndOwner  = hwnd;
          cf.lpLogFont  = &glf;
          cf.iPointSize = gFontSize;
          cf.lpszStyle  = gFontName;
          cf.Flags      = CF_ANSIONLY | CF_SCREENFONTS | CF_INITTOLOGFONTSTRUCT;
          cf.rgbColors  = fgtext_color;//RGB(0, 255, 255); /* light blue */
          cf.nFontType  = SCREEN_FONTTYPE;
          ChooseFont(&cf);
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDC_TEXT_FONT:
          memset(&cf, 0, sizeof(CHOOSEFONT));
          cf.lStructSize= sizeof(CHOOSEFONT);
          cf.hwndOwner  = hwnd;
          cf.lpLogFont  = &tlf;
          cf.iPointSize = tFontSize;
          cf.lpszStyle  = tFontName;
          cf.Flags      = CF_FIXEDPITCHONLY | CF_ANSIONLY | CF_SCREENFONTS | CF_INITTOLOGFONTSTRUCT;
          cf.nFontType  = SCREEN_FONTTYPE;
          ChooseFont(&cf);
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;
      }
      break;

    case WM_NOTIFY:
      lpnmhdr = (NMHDR FAR *)lParam;

      switch (lpnmhdr->code){

        case PSN_HELP:
          WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Preferences - Colors and Fonts"));
          break;

        case PSN_APPLY:   //sent when OK or Apply button pressed
          ResetPen(&bgday_color_gc,   &bgday_color,   hwnd,IDC_BGDAY);
          ResetPen(&bgnite_color_gc,  &bgnite_color,  hwnd,IDC_BGNITE);
          ResetPen(&fgtext_color_gc,  &fgtext_color,  hwnd,IDC_FGTEXT);
          ResetPen(&fgrise_color_gc,  &fgrise_color,  hwnd,IDC_FGRISE);
          ResetPen(&fgfall_color_gc,  &fgfall_color,  hwnd,IDC_FGFALL);
          ResetPen(&fgmiddle_color_gc,&fgmiddle_color,hwnd,IDC_FGMIDDLE);
          ResetPen(&fgmark_color_gc,  &fgmark_color,  hwnd,IDC_FGMARK);
          ResetPen(&fgmllw_color_gc,  &fgmllw_color,  hwnd,IDC_FGMLLW);
          ResetPen(&fgmapdot_color_gc,&fgmapdot_color,hwnd,IDC_FGTDOT);
          ResetPen(&fgcurdot_color_gc,&fgcurdot_color,hwnd,IDC_FGCDOT);
          ptSize = (int)((float)glf.lfHeight*-72L / (float)GetDeviceCaps(hDcDisplay, LOGPIXELSY)+ 0.5);
          if (ptSize != gFontSize || strcmp(gFontName,glf.lfFaceName)) {
             strcpy(gFontName, glf.lfFaceName);
             gFontSize = ptSize;
             new_params = TRUE;
             DeleteObject(gfont);
             gfont = NULL;
          }
          ptSize = (int)((float)tlf.lfHeight*-72L / (float)GetDeviceCaps(hDcDisplay, LOGPIXELSY)+ 0.5);
          if (ptSize != tFontSize || strcmp(tFontName,tlf.lfFaceName)) {
             strcpy(tFontName, tlf.lfFaceName);
             tFontSize = ptSize;
             new_params = TRUE;
             DeleteObject(tfont);
             tfont = NULL;
             if (hwndText!=NULL)
                printf("");
//                MessageBox(hwnd, "Text window font change will not take effect\r\n"
//                                 "until the text window has been closed.",
//                                 "Text window font", MB_ICONINFORMATION);
          }
          break;

//        case PSN_RESET:   //sent when Cancel button pressed
//          break;

//        case PSN_SETACTIVE:
//          break;

        default:
          break;
      }
    break;
  }
  ReleaseDC(hwnd, hDcDisplay);
  return FALSE;
}


/*-----------------12/17/2005 1:34PM----------------
 * Preferences Sheet General page
 * --------------------------------------------------*/
BOOL CALLBACK PSGeneralDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam) {
static int units_m, new_units, new_tz, tmpInt, was_tadj;
int  gotIt;
LPNMHDR    lpnmhdr;

  switch(msg) {

    case WM_INITDIALOG:
      units_m = ((youwant) && tolower( youwant[0] == 'm'));
      CheckDlgButton(hwnd,IDG_CONVFM,( youwant!=NULL));
      CheckDlgButton(hwnd,IDG_FEET  ,( youwant && !units_m));
      CheckDlgButton(hwnd,IDG_METERS,( youwant &&  units_m));
      CheckDlgButton(hwnd,IDG_LOCALTZ,(!loctz && !Usetadjust && !utc ));
      CheckDlgButton(hwnd,IDG_UTC,       utc  && !Usetadjust && !loctz);
      CheckDlgButton(hwnd,IDG_REMTZ,    loctz && !Usetadjust && !utc);
      CheckDlgButton(hwnd,IDG_FIXEDTZ,  Usetadjust);
      was_tadj = Usetadjust;
      LoadTZCombo(   hwnd,IDG_TZNAME);
      CheckDlgButton(hwnd,IDG_YMD,   (datemdy ==0));
      CheckDlgButton(hwnd,IDG_MDY,   (datemdy !=0));
      CheckDlgButton(hwnd,IDG_CUES,  (map_cues >0));
      CheckDlgButton(hwnd,IDG_CUESH, (map_cues==1));
      CheckDlgButton(hwnd,IDG_CUESF, (map_cues==2));
      CheckDlgButton(hwnd,IDG_ONESCN,(OnESCape<=0));
      CheckDlgButton(hwnd,IDG_ONESCM,(OnESCape==1));
      CheckDlgButton(hwnd,IDG_ONESCX,(OnESCape==2));
      CheckDlgButton(hwnd,IDG_ICON,  (UseIcon  >0));
      CheckDlgButton(hwnd,IDG_ICONM, (UseIcon ==1));
      CheckDlgButton(hwnd,IDG_ICONA, (UseIcon ==2));
      CheckDlgButton(hwnd,IDG_AUTOKT, convert_BOGUS);
      CheckDlgButton(hwnd,IDG_UPDATE, auto_save);
      CheckDlgButton(hwnd,IDG_ONLY1,  OnlyOne);
      SetDlgItemInt(hwnd,IDG_MAX_MRU, (UINT)max_mru, FALSE);
      SetDlgItemInt(hwnd,IDG_SUBP,    (UINT)subproc, FALSE);
      break;

    case WM_COMMAND:
      switch(LOWORD(wParam)) {

        case IDG_CONVFM:
          tmpInt = IsDlgButtonChecked(hwnd,IDG_CONVFM);
          CheckDlgButton(hwnd,IDG_FEET  ,(tmpInt && !units_m));
          CheckDlgButton(hwnd,IDG_METERS,(tmpInt &&  units_m));
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_FEET:
          if (!(youwant && !units_m)) {
            CheckDlgButton(hwnd,IDG_CONVFM,TRUE);
            CheckDlgButton(hwnd,IDG_FEET  ,TRUE);
            CheckDlgButton(hwnd,IDG_METERS,FALSE);
            PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDG_METERS:
          if (!(youwant && units_m)) {
            CheckDlgButton(hwnd,IDG_CONVFM,TRUE );
            CheckDlgButton(hwnd,IDG_FEET  ,FALSE);
            CheckDlgButton(hwnd,IDG_METERS,TRUE );
            PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDG_LOCALTZ:
          if (!(!loctz && !Usetadjust && !utc ))
            PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_UTC:
          if (!(utc  && !Usetadjust && !loctz))
            PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_REMTZ:
          if (!(loctz && !Usetadjust && !utc))
            PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_FIXEDTZ:
          if (!Usetadjust)
            PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_TZNAME:
          if ((HIWORD(wParam)) == CBN_KILLFOCUS) {
            if (GetTZCombo(hwnd, IDG_TZNAME, IDG_FIXEDTZ, 1))
              PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDG_YMD:
          if (datemdy) {
            PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDG_MDY:
          if (!datemdy) {
            PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDG_CUES:
          tmpInt = IsDlgButtonChecked(hwnd,IDG_CUES);
          CheckDlgButton(hwnd,IDG_CUESH, (tmpInt && map_cues!=2));
          CheckDlgButton(hwnd,IDG_CUESF, (tmpInt && map_cues==2));
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_CUESH:
          if (map_cues!=1) {
            CheckDlgButton(hwnd,IDG_CUES, TRUE);
            PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDG_CUESF:
          if (map_cues!=2) {
            CheckDlgButton(hwnd,IDG_CUES, TRUE);
            PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDG_ONESCN:
          if (OnESCape!=0)
            PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_ONESCM:
          if (OnESCape!=1)
            PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_ONESCX:
          if (OnESCape!=2)
            PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_ICON:
          tmpInt = IsDlgButtonChecked(hwnd,IDG_ICON);
          CheckDlgButton(hwnd,IDG_ICONM, (tmpInt && UseIcon!=2));
          CheckDlgButton(hwnd,IDG_ICONA, (tmpInt && UseIcon==2));
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDG_ICONM:
          if (UseIcon!=1) {
            CheckDlgButton(hwnd,IDG_ICON, TRUE);
            PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDG_ICONA:
          if (UseIcon!=2) {
            CheckDlgButton(hwnd,IDG_ICON, TRUE);
            PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDG_MAX_MRU:
          if ((HIWORD(wParam)) == EN_KILLFOCUS) {
            tmpInt = GetDlgItemInt(hwnd,IDG_MAX_MRU, &gotIt, FALSE);
            if (gotIt && tmpInt >= 0 && tmpInt <= 9) {
              if (tmpInt != max_mru)
                PropSheet_Changed(GetParent(hwnd), hwnd);
            }
            else
              SetDlgItemInt(hwnd,IDG_MAX_MRU, (UINT)max_mru, FALSE);
          }
          break;

        case IDG_SUBP:
          if ((HIWORD(wParam)) == EN_KILLFOCUS) {
            tmpInt = GetDlgItemInt(hwnd,IDG_SUBP, &gotIt, FALSE);
            if (gotIt && tmpInt >= 0 && tmpInt <= 3) {
              if (tmpInt != subproc)
                PropSheet_Changed(GetParent(hwnd), hwnd);
            }
            else
              SetDlgItemInt(hwnd,IDG_SUBP, (UINT)subproc, FALSE);
          }
          break;


        case IDG_UPDATE:
        case IDG_ONLY1:
          PropSheet_Changed(GetParent(hwnd), hwnd);

        default:
          break;
      }
      break;

    case WM_NOTIFY:
      lpnmhdr = (NMHDR FAR *)lParam;

      switch (lpnmhdr->code){

        case PSN_HELP:
          WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Preferences - System"));
          break;

        case PSN_APPLY:   //sent when OK or Apply button pressed
          new_tz  = GetTZCombo(hwnd, IDG_TZNAME, IDG_FIXEDTZ,0);
          new_tz |= was_tadj != Usetadjust;
          new_tz |= get_flag(hwnd,&utc,  IDG_UTC);
          new_tz |= get_flag(hwnd,&loctz,IDG_REMTZ);

          get_flag(hwnd,&auto_save,     IDG_UPDATE);
          get_flag(hwnd,&datemdy,       IDG_MDY);

          if      ( IsDlgButtonChecked(hwnd,IDG_ONESCN)) tmpInt = 0;
          else if ( IsDlgButtonChecked(hwnd,IDG_ONESCM)) tmpInt = 1;
          else                                           tmpInt = 2;
          if (OnESCape != tmpInt) {
            OnESCape   = tmpInt;
            new_params = TRUE;
          }

          if      (!IsDlgButtonChecked(hwnd,IDG_ICON )) tmpInt = 0;
          else if ( IsDlgButtonChecked(hwnd,IDG_ICONM)) tmpInt = 1;
          else                                          tmpInt = 2;
          if (UseIcon != tmpInt) {
            UseIcon    = tmpInt;
            new_params = TRUE;
            ForceUpdateIcon = 1;
          }

          if      (!IsDlgButtonChecked(hwnd,IDG_CUES )) tmpInt = 0;
          else if ( IsDlgButtonChecked(hwnd,IDG_CUESH)) tmpInt = 1;
          else                                          tmpInt = 2;
          if (map_cues != tmpInt) {
            map_cues   = tmpInt;
            new_params = TRUE;
          }

          tmpInt = GetDlgItemInt(hwnd,IDG_MAX_MRU, &gotIt, FALSE);
          if (max_mru!=tmpInt) new_params = TRUE;
          max_mru = tmpInt;

          tmpInt = GetDlgItemInt(hwnd,IDG_SUBP, &gotIt, FALSE);
          if (subproc!=tmpInt) new_params = TRUE;
          subproc = tmpInt;

          units_m   = ( youwant &&  tolower(youwant[0] == 'm'));
          new_units = ( youwant &&  units_m && !IsDlgButtonChecked(hwnd,IDG_METERS)) ||
                      ( youwant && !units_m && !IsDlgButtonChecked(hwnd,IDG_FEET)) ||
                      (!youwant && (IsDlgButtonChecked(hwnd,IDG_FEET)||IsDlgButtonChecked(hwnd,IDG_METERS)));
          if (new_units) {
            if (IsDlgButtonChecked(hwnd,IDG_FEET))
              strADupe(&youwant, "ft");
            else if (IsDlgButtonChecked(hwnd,IDG_METERS))
              strADupe(&youwant, "m");
            else youwant = NULL;
          }
          tmpInt = convert_BOGUS;
          convert_BOGUS = IsDlgButtonChecked(hwnd,IDG_AUTOKT) == BST_CHECKED? 1 : 0;
          if (new_units || (convert_BOGUS != tmpInt) || new_tz) {
            new_params = TRUE;
            load_location_data(IDX_station_name, IDX_rec_num);
          }
          section_save();
          if (get_flag(hwnd,&OnlyOne,IDG_ONLY1) && !auto_save) {
            if (IDYES==MessageBox(hwnd,
                "Changes to the Only one instance option will not\ngo into effect until the configuration is saved.\n\nSave now?",
                "Only one instance option",MB_YESNO | MB_ICONEXCLAMATION)) {
              write_config_file(szConfigPath, TRUE);
              WinTideCheckMenus(hwnd);
            }
          }
          InvalidateRect(hwndMain,NULL,TRUE);
          UpdateWindow(hwndMain);
          break;

//        case PSN_RESET:   //sent when Cancel button pressed
//          break;

//        case PSN_SETACTIVE:
//          break;

        default:
          break;
      }
    break;
  }
  return FALSE;
}


/*-----------------12/17/2005 1:34PM----------------
 * Preferences Sheet Modes page
 * --------------------------------------------------*/
BOOL CALLBACK PSModesDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam) {
int      scrollsave;
float    tmpFloat;
LPNMHDR  lpnmhdr;
PROPSHEETPAGE *psp;
static int pageSelected;
static section_entry *pSection;

  switch(msg) {
    case WM_INITDIALOG:
      psp = (PROPSHEETPAGE *)lParam;
      pageSelected = psp->lParam;
//printf("\nInit %d", pageSelected);
      if (pageSelected < 0 || pageSelected > 3) pageSelected = 0;
      if (pageSelected == section_what())
        section_write(pageSelected);
      pSection = &section[pageSelected];
      SetDlgItemInt(hwnd, IDF_NAME, pageSelected,FALSE);
      CheckDlgButton(hwnd,IDF_SHOWTICS,pSection->lines);
      if (!pSection->lines) {
        CheckDlgButton(hwnd,IDF_HINC,      BST_INDETERMINATE);
        CheckDlgButton(hwnd,IDF_TINC,      BST_INDETERMINATE);
        CheckDlgButton(hwnd,IDF_CALIB,     BST_INDETERMINATE);
        CheckDlgButton(hwnd,IDF_DEPTHLINES,BST_INDETERMINATE);
        CheckDlgButton(hwnd,IDF_TOPLINES,  BST_INDETERMINATE);
      }
      else {
        CheckDlgButton(hwnd,IDF_HINC,      pSection->hinc?           BST_CHECKED : BST_UNCHECKED);
        CheckDlgButton(hwnd,IDF_TINC,      pSection->tinc?           BST_CHECKED : BST_UNCHECKED);
        CheckDlgButton(hwnd,IDF_CALIB,     pSection->hairycalibrate? BST_CHECKED : BST_UNCHECKED);
        CheckDlgButton(hwnd,IDF_DEPTHLINES,pSection->depthlines?     BST_CHECKED : BST_UNCHECKED);
        if (pSection->depthlines)
             CheckDlgButton(hwnd,IDF_TOPLINES,pSection->toplines? BST_CHECKED : BST_UNCHECKED);
        else CheckDlgButton(hwnd,IDF_TOPLINES,BST_INDETERMINATE);
      }
      CheckDlgButton(hwnd,IDF_NOW,    pSection->now);
      CheckDlgButton(hwnd,IDF_24,     pSection->noampm);
      CheckDlgButton(hwnd,IDF_AMPL,   pSection->showampl);
      CheckDlgButton(hwnd,IDF_MIDDLE, pSection->middle);
      CheckDlgButton(hwnd,IDF_MLLW,   pSection->mllw);
      CheckDlgButton(hwnd,IDF_MARK,   pSection->mark);
      CheckDlgButton(hwnd,IDF_SKINNY, pSection->skinny == 2);
      CheckDlgButton(hwnd,IDF_THIN,   pSection->skinny == 1);
      CheckDlgButton(hwnd,IDF_FULL,   pSection->skinny == 0);
      CheckDlgButton(hwnd,IDF_MOON,   pSection->show_moons);
      CheckDlgButton(hwnd,IDF_NOFILL, pSection->linegraph);
      CheckDlgButton(hwnd,IDF_SCROLL, pSection->usescrollbar);
      CheckDlgButton(hwnd,IDF_CAPTION,pSection->caption);
      CheckDlgButton(hwnd,IDF_STAGGER,pSection->stagger);
      CheckDlgButton(hwnd,IDF_WEEK,   pSection->weekday);
      if (pSection->stagger)
           CheckDlgButton(hwnd,IDF_STAGGERFLOAT,pSection->staggerfloat? BST_CHECKED : BST_UNCHECKED);
      else CheckDlgButton(hwnd,IDF_STAGGERFLOAT,BST_INDETERMINATE);
      SendDlgItemMessage(hwnd, IDF_MARKLEV, EM_SETREADONLY, !pSection->mark, 0L);
      CheckDlgButton(hwnd,IDF_ONEDATE,pSection->onedate);
      SetDlgItemText(hwnd,IDF_STRETCH, makeRealStr(180.0/pSection->tstep));
      SetDlgItemText(hwnd,IDF_MARKLEV, makeRealStr(pSection->marklev));
      break;

    case WM_COMMAND:
      switch(LOWORD(wParam)) {

        case IDF_SHOWTICS:
          if (IsDlgButtonChecked(hwnd,IDF_SHOWTICS)) {
            CheckDlgButton(hwnd,IDF_HINC,      pSection->hinc?          BST_CHECKED : BST_UNCHECKED);
            CheckDlgButton(hwnd,IDF_TINC,      pSection->tinc?          BST_CHECKED : BST_UNCHECKED);
            CheckDlgButton(hwnd,IDF_CALIB,     pSection->hairycalibrate?BST_CHECKED : BST_UNCHECKED);
            CheckDlgButton(hwnd,IDF_DEPTHLINES,pSection->depthlines?    BST_CHECKED : BST_UNCHECKED);
            if (IsDlgButtonChecked(hwnd,IDF_DEPTHLINES))
                 CheckDlgButton(hwnd,IDF_TOPLINES,pSection->toplines? BST_CHECKED : BST_UNCHECKED);
            else CheckDlgButton(hwnd,IDF_TOPLINES,BST_INDETERMINATE);
          }
          else {
            CheckDlgButton(hwnd,IDF_HINC,      BST_INDETERMINATE);
            CheckDlgButton(hwnd,IDF_TINC,      BST_INDETERMINATE);
            CheckDlgButton(hwnd,IDF_CALIB,     BST_INDETERMINATE);
            CheckDlgButton(hwnd,IDF_DEPTHLINES,BST_INDETERMINATE);
            CheckDlgButton(hwnd,IDF_TOPLINES,  BST_INDETERMINATE);
          }
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDF_HINC:
        case IDF_TINC:
        case IDF_CALIB:
          if (IsDlgButtonChecked(hwnd,IDF_SHOWTICS))
            CheckDlgButton(hwnd,wParam,
                IsDlgButtonChecked(hwnd,wParam)? BST_UNCHECKED:BST_CHECKED);
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDF_DEPTHLINES:
          if (IsDlgButtonChecked(hwnd,IDF_SHOWTICS))
            CheckDlgButton(hwnd,IDF_DEPTHLINES,
                IsDlgButtonChecked(hwnd,IDF_DEPTHLINES)? BST_UNCHECKED:BST_CHECKED);
          if (IsDlgButtonChecked(hwnd,IDF_DEPTHLINES)&&IsDlgButtonChecked(hwnd,IDF_SHOWTICS))
            CheckDlgButton(hwnd,IDF_TOPLINES,pSection->toplines? BST_CHECKED : BST_UNCHECKED);
          else
            CheckDlgButton(hwnd,IDF_TOPLINES, BST_INDETERMINATE);
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDF_TOPLINES:
          if (IsDlgButtonChecked(hwnd,IDF_DEPTHLINES)&&IsDlgButtonChecked(hwnd,IDF_SHOWTICS))
            CheckDlgButton(hwnd,IDF_TOPLINES,
                IsDlgButtonChecked(hwnd,IDF_TOPLINES)? BST_UNCHECKED:BST_CHECKED);
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDF_STAGGER:
          if (IsDlgButtonChecked(hwnd,IDF_STAGGER))
            CheckDlgButton(hwnd,IDF_STAGGERFLOAT,pSection->staggerfloat? BST_CHECKED : BST_UNCHECKED);
          else
            CheckDlgButton(hwnd,IDF_STAGGERFLOAT, BST_INDETERMINATE);
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDF_STAGGERFLOAT:
          if (IsDlgButtonChecked(hwnd,IDF_STAGGER))
            CheckDlgButton(hwnd,IDF_STAGGERFLOAT,
                IsDlgButtonChecked(hwnd,IDF_STAGGERFLOAT)? BST_UNCHECKED:BST_CHECKED);
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDF_MARK:
          SendDlgItemMessage(hwnd, IDF_MARKLEV, EM_SETREADONLY,
                !IsDlgButtonChecked(hwnd,IDF_MARK), 0L);
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;

        case IDF_STRETCH:
          if ((HIWORD(wParam)) == EN_KILLFOCUS) {
            if (GetDlgItemFloat(hwnd, IDF_STRETCH, &tmpFloat) &&  // value is numeric
                tmpFloat >= 0.01 && tmpFloat < 10.0           &&  // ... and it's legal
                !FSAME((180.0/tmpFloat), pSection->tstep)       ) // ... and has changed
              PropSheet_Changed(GetParent(hwnd), hwnd);
            else                                            // Else restore previous value
              SetDlgItemText(hwnd,IDF_STRETCH, makeRealStr(180.0 / pSection->tstep));
          }
          break;

        case IDF_MARKLEV:
          if ((HIWORD(wParam)) == EN_KILLFOCUS) {
            if (GetDlgItemFloat(hwnd, IDF_MARKLEV, &tmpFloat) &&
                !FSAME(tmpFloat, pSection->marklev))
              PropSheet_Changed(GetParent(hwnd), hwnd);
            else
              SetDlgItemText(hwnd,IDF_MARKLEV, makeRealStr(pSection->marklev));
          }
          break;

        case IDF_FULL :
        case IDF_THIN :
        case IDF_SKINNY :
          if (!IsDlgButtonChecked(hwnd, LOWORD(wParam))) {
            CheckDlgButton(hwnd,IDF_SKINNY,IDF_SKINNY==LOWORD(wParam)? TRUE : FALSE);
            CheckDlgButton(hwnd,IDF_THIN,  IDF_THIN  ==LOWORD(wParam)? TRUE : FALSE);
            CheckDlgButton(hwnd,IDF_FULL,  IDF_FULL  ==LOWORD(wParam)? TRUE : FALSE);
            PropSheet_Changed(GetParent(hwnd), hwnd);
          }
          break;

        case IDF_NOW    :
        case IDF_24     :
        case IDF_MIDDLE :
        case IDF_MLLW   :
        case IDF_AMPL   :
        case IDF_MOON   :
        case IDF_NOFILL :
        case IDF_SCROLL :
        case IDF_ONEDATE:
        case IDF_CAPTION:
        case IDF_WEEK   :
          PropSheet_Changed(GetParent(hwnd), hwnd);
          break;
      }
      break;

    case WM_NOTIFY:
      lpnmhdr = (NMHDR FAR *)lParam;

      switch (lpnmhdr->code){

        case PSN_HELP:
          WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Preferences - Mode specific"));
          break;

        case PSN_APPLY:   //sent when OK or Apply button pressed
          pageSelected = GetDlgItemInt(hwnd, IDF_NAME, NULL,FALSE);
//printf("Apply %d", pageSelected);
          if (pageSelected < 0 || pageSelected > 3) pageSelected = 0;
          pSection = &section[pageSelected];
          get_flag(hwnd,&pSection->lines,          IDF_SHOWTICS);
          if (pSection->lines) {
            get_3st(hwnd,&pSection->hinc,          IDF_HINC);
            if (!pSection->hinc) hincmagic = 0;
            get_3st(hwnd,&pSection->tinc,          IDF_TINC);
            get_3st(hwnd,&pSection->hairycalibrate,IDF_CALIB);
            get_3st(hwnd,&pSection->depthlines,    IDF_DEPTHLINES);
            if (pSection->depthlines)
              get_3st(hwnd,&pSection->toplines,    IDF_TOPLINES);
          }
          get_flag(hwnd,&pSection->now,            IDF_NOW);
          get_flag(hwnd,&pSection->noampm,         IDF_24);

          get_flag(hwnd,&pSection->showampl,       IDF_AMPL);
          get_flag(hwnd,&pSection->middle,         IDF_MIDDLE);
          get_flag(hwnd,&pSection->mllw,           IDF_MLLW);
          get_flag(hwnd,&pSection->stagger,        IDF_STAGGER);
          if (pSection->stagger)
            get_3st(hwnd,&pSection->staggerfloat,  IDF_STAGGERFLOAT);
          get_flag(hwnd,&pSection->mark,           IDF_MARK);
          get_flag(hwnd,&pSection->show_moons,     IDF_MOON);
          get_flag(hwnd,&pSection->linegraph,      IDF_NOFILL);
          get_flag(hwnd,&pSection->onedate,        IDF_ONEDATE);
          get_flag(hwnd,&pSection->caption,        IDF_CAPTION);
          get_flag(hwnd,&pSection->weekday,        IDF_WEEK);

          if (pageSelected == MODE_GRAPH || pageSelected == MODE_OVERVIEW) {
            scrollsave = pSection->usescrollbar;
            get_flag(hwnd,&pSection->usescrollbar,   IDF_SCROLL);
            if ((scrollsave != pSection->usescrollbar) && pageSelected == section_what()) {//graphmode) {
              usescrollbar = pSection->usescrollbar;
              DoScrollBar(TRUE);
              InvalidateRect(hwndMain,NULL,TRUE);
            }
          }
          if  (IsDlgButtonChecked(hwnd,IDF_SKINNY)) {
            if (pSection->skinny!=2) new_params = TRUE;
            pSection->skinny = 2;
          }
          else if (IsDlgButtonChecked(hwnd,IDF_THIN)) {
            if (pSection->skinny!=1) new_params = TRUE;
            pSection->skinny = 1;
          }
          else {
            if (pSection->skinny) new_params = TRUE;
            pSection->skinny = 0;
          }

          if (GetDlgItemFloat(hwnd, IDF_MARKLEV, &tmpFloat) &&
             !FSAME(pSection->marklev, tmpFloat)) {
            new_params = TRUE;
            pSection->marklev = tmpFloat;
          }

          if (GetDlgItemFloat(hwnd, IDF_STRETCH, &tmpFloat) &&
              tmpFloat >= 0.01 && tmpFloat < 10.0  &&
              !FSAME((180.0/tmpFloat), pSection->tstep)) {
            new_params = TRUE;
            pSection->tstep = (int)(180.0 / tmpFloat);
          }

          if (pageSelected == section_what()) {
            section_read(pageSelected);
            InvalidateRect(hwndMain,NULL,TRUE);
            UpdateWindow(hwndMain);
          }
          break;

//        case PSN_RESET:   //sent when Cancel button pressed
//          break;

        case PSN_SETACTIVE:
          pageSelected = GetDlgItemInt(hwnd, IDF_NAME, NULL,FALSE);
          if (pageSelected < 0 || pageSelected > 3) pageSelected = 0;
//printf("SetActive %d", pageSelected);
          break;

        default:
          break;
      }
    break;
  }
  return FALSE;
}


/**************************************************************************

   PropSheetCallback()

**************************************************************************/

void CALLBACK PropSheetCallback(HWND hwndPropSheet, UINT uMsg, LPARAM lParam)
{
  switch(uMsg){
  //called before the dialog is created,
  //hwndPropSheet = NULL, lParam points to dialog resource
  case PSCB_PRECREATE:{
    LPDLGTEMPLATE  lpTemplate = (LPDLGTEMPLATE)lParam;

    if(!(lpTemplate->style & WS_SYSMENU)){
      lpTemplate->style |= WS_SYSMENU;
    }
  }
  break;

  //called after the dialog is created
  case PSCB_INITIALIZED:
    break;
  }
}


/**************************************************************************

   DoModalPropSheet()

**************************************************************************/

INT_PTR PrefsPropSheet( HWND hwndOwner )
{
int modeToPageMap[4] = {2, 3, 5, 4};
  PROPSHEETPAGE psp[6];
  PROPSHEETHEADER psh;
//          if(!DialogBox(g_hinst,"Flags",hwnd,FlagDlg))

  //Fill out the PROPSHEETPAGE data structure for the General sheet
  psp[0].dwSize      = sizeof(PROPSHEETPAGE);
  psp[0].dwFlags     = PSP_USETITLE | PSP_HASHELP;
  psp[0].hInstance   = g_hinst;
  psp[0].pszTemplate = "GENERAL_PAGE";
  psp[0].pszIcon     = NULL;
  psp[0].pfnDlgProc  = PSGeneralDlg;
  psp[0].pszTitle    = TEXT("System");
  psp[0].lParam      = 0;

  //Fill out the PROPSHEETPAGE data structure for the Colors and Fonts sheet
  psp[1].dwSize      = sizeof(PROPSHEETPAGE);
  psp[1].dwFlags     = PSP_USETITLE | PSP_HASHELP;
  psp[1].hInstance   = g_hinst;
  psp[1].pszTemplate = "COLOR_PAGE";
  psp[1].pszIcon     = NULL;
  psp[1].pfnDlgProc  = ColorDlg;
  psp[1].pszTitle    = TEXT("Colors && Fonts");
  psp[1].lParam      = 0;

  //Fill out the PROPSHEETPAGE data structure for the Hairy Mode sheet
  psp[2].dwSize      = sizeof(PROPSHEETPAGE);
  psp[2].dwFlags     = PSP_USETITLE | PSP_HASHELP;
  psp[2].hInstance   = g_hinst;
  psp[2].pszTemplate = "HAIRY_FLAGS_PAGE";
  psp[2].pszIcon     = NULL;
  psp[2].pfnDlgProc  = PSModesDlg;
  psp[2].pszTitle    = TEXT("Realtime");
  psp[2].lParam      = MODE_HAIRY;

  //Fill out the PROPSHEETPAGE data structure for the Graph Mode sheet
  psp[3].dwSize      = sizeof(PROPSHEETPAGE);
  psp[3].dwFlags     = PSP_USETITLE | PSP_HASHELP;
  psp[3].hInstance   = g_hinst;
  psp[3].pszTemplate = "GRAPH_FLAGS_PAGE";
  psp[3].pszIcon     = NULL;
  psp[3].pfnDlgProc  = PSModesDlg;
  psp[3].pszTitle    = TEXT("Graph");
  psp[3].lParam      = MODE_GRAPH;

  //Fill out the PROPSHEETPAGE data structure for the Overview Mode sheet
  psp[4].dwSize      = sizeof(PROPSHEETPAGE);
  psp[4].dwFlags     = PSP_USETITLE | PSP_HASHELP;
  psp[4].hInstance   = g_hinst;
  psp[4].pszTemplate = "GRAPH_FLAGS_PAGE";
  psp[4].pszIcon     = NULL;
  psp[4].pfnDlgProc  = PSModesDlg;
  psp[4].pszTitle    = TEXT("Overview");
  psp[4].lParam      = MODE_OVERVIEW;

  //Fill out the PROPSHEETPAGE data structure for the Clock Mode sheet
  psp[5].dwSize      = sizeof(PROPSHEETPAGE);
  psp[5].dwFlags     = PSP_USETITLE | PSP_HASHELP;
  psp[5].hInstance   = g_hinst;
  psp[5].pszTemplate = "CLOCK_FLAGS_PAGE";//MAKEINTRESOURCE(IDD_BUTTONS);
  psp[5].pszIcon     = NULL;
  psp[5].pfnDlgProc  = PSModesDlg;
  psp[5].pszTitle    = TEXT("Clock");
  psp[5].lParam      = MODE_CLOCK;

  //Fill out the PROPSHEETHEADER
  psh.dwSize         = sizeof(PROPSHEETHEADER);
  psh.dwFlags        = PSH_PROPSHEETPAGE | PSH_USEICONID | PSH_USECALLBACK | PSH_HASHELP;// | PSH_PROPTITLE  ;
  psh.hwndParent     = hwndOwner;
  psh.hInstance      = g_hinst;
  psh.pszIcon        = NULL;//MAKEINTRESOURCE(IDI_BACKCOLOR);
  psh.pszCaption     = TEXT("Preferences");
  psh.nPages         = sizeof(psp) / sizeof(PROPSHEETPAGE);
  psh.nStartPage     = modeToPageMap[section_what() %4];  // Start on page for this mode
  psh.ppsp           = (LPCPROPSHEETPAGE) &psp;
  psh.pfnCallback    = (PFNPROPSHEETCALLBACK)PropSheetCallback;

  //And finally display the property sheet
  return PropertySheet(&psh);
}


BOOL CALLBACK DateCalendarDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam) {
HWND hCal;
SYSTEMTIME stToday, stMinMax[2]; // Today's date and min/max array
struct tm ttm;
time_t t_gmt, t_local;
//#define MCN_SELECT MCN_FIRST+4

  hCal = GetDlgItem(hwnd, XID_CALENDAR); // Get handle to MonthCalendar control
  switch(msg) {
    case WM_INITDIALOG:
      GetLocalTime(&stToday);
      stMinMax[0].wYear  = first_year;
      stMinMax[0].wMonth = 1;
      stMinMax[0].wDay   = 1;
      stMinMax[1].wYear  = first_year+num_epochs-1;
      stMinMax[1].wMonth = 12;
      stMinMax[1].wDay   = 30;
      stMinMax[0].wDayOfWeek = stMinMax[1].wDayOfWeek = 0;
      stMinMax[0].wHour      = stMinMax[1].wHour      = 0;
      stMinMax[0].wMinute    = stMinMax[1].wMinute    = 0;
      stMinMax[0].wSecond    = stMinMax[1].wSecond    = 0;
      stMinMax[0].wMilliseconds = stMinMax[1].wMilliseconds = 0;
      MonthCal_SetCurSel(hCal, &stToday);
      MonthCal_SetRange (hCal, GDTR_MIN | GDTR_MAX, &stMinMax);
      SetFocus(hCal);
      ReleaseCapture();
//      PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, IDOK), -1);
      return 0;

    case WM_CLOSE:
      PostQuitMessage(0);
      return 0;

    case WM_NOTIFY:
      if ( ((int)wParam == XID_CALENDAR) && (((LPNMHDR) lParam)->code == MCN_SELECT) ) {
          PostMessage(hwnd,WM_COMMAND,IDOK,0L);
          return(0);
      }

    case WM_COMMAND:
      switch(LOWORD(wParam)) {
        case IDOK:
          // Get currently selected date from MC control
          MonthCal_GetCurSel(hCal, &stToday);
          ttm.tm_year = stToday.wYear - 1900;
          ttm.tm_mon  = stToday.wMonth - 1;
          ttm.tm_mday = stToday.wDay;
          ttm.tm_hour = ttm.tm_min = ttm.tm_sec = ttm.tm_isdst = 0;
          t_gmt = tm2gmt( &ttm );
          t_local = tz_gm2localtime(t_gmt);
          gstart_time = t_gmt + (t_gmt - t_local);
          PostQuitMessage(1);
          return(0);

        case IDCANCEL:
          PostQuitMessage(0);
          return(0);
      }
  }
  return 0;
}

int DateTimePicker(HWND hwnd) {
MSG msg;
HWND hwndDlg = NULL;
  if ((hwndDlg = CreateDialog(g_hinst, "DateCal", hwnd, DateCalendarDlg)) != NULL) {

    ShowWindow(hwndDlg,SW_SHOW);
    {
    HACCEL hDTAccel;
      hDTAccel=LoadAccelerators(g_hinst,"DATETIME");
      while(GetMessage(&msg,NULL,0,0)) {
        if(!TranslateAccelerator(hwndDlg,hDTAccel,&msg)) {
          TranslateMessage(&msg);
          DispatchMessage(&msg);
        }
      }
    }
    if (hwndDlg) DestroyWindow(hwndDlg);
    SetFocus(hwndMain);
    return msg.wParam;
  }
  else printf("The Date Picker Common Control cannot be intitialized.\n"
              "It requires comctl32.dll version 4.70 or later.");
  return(0);
}

static int TextOption;
/*-----------------1/6/2003 8:29PM------------------
 *
 * --------------------------------------------------*/
void set_text_options(HWND hwnd) {
char tmpStr[60];
int skinny_save;
  if (TextOption == ID_CALENDAR) {
    CheckDlgButton(    hwnd, ID_TIDES,    FALSE);
    CheckDlgButton(    hwnd, ID_CALENDAR, TRUE );
    CheckDlgButton(    hwnd, ID_INCREMENT,FALSE);
    CheckDlgButton(    hwnd, IDT_SUNMOON, BST_INDETERMINATE);
    SendDlgItemMessage(hwnd, IDT_MINUTES, EM_SETREADONLY, TRUE,  0L);
    SendDlgItemMessage(hwnd, IDT_DAYS,    EM_SETREADONLY, TRUE,  0L);
    SendDlgItemMessage(hwnd, IDT_MONTHS,  EM_SETREADONLY, FALSE, 0L);
  }
  else if (TextOption == ID_TIDES) {
    CheckDlgButton(    hwnd, ID_TIDES,    TRUE );
    CheckDlgButton(    hwnd, ID_CALENDAR, FALSE);
    CheckDlgButton(    hwnd, ID_INCREMENT,FALSE);
    CheckDlgButton(    hwnd, IDT_SUNMOON, sun_moon? BST_CHECKED : BST_UNCHECKED);
    SendDlgItemMessage(hwnd, IDT_MINUTES, EM_SETREADONLY, TRUE , 0L);
    SendDlgItemMessage(hwnd, IDT_DAYS,    EM_SETREADONLY, FALSE, 0L);
    SendDlgItemMessage(hwnd, IDT_MONTHS,  EM_SETREADONLY, TRUE , 0L);
  }
  else /* (TextOption == ID_INCREMENT)*/ {
    CheckDlgButton(    hwnd, ID_TIDES,    FALSE);
    CheckDlgButton(    hwnd, ID_CALENDAR, FALSE);
    CheckDlgButton(    hwnd, ID_INCREMENT,TRUE );
    CheckDlgButton(    hwnd, IDT_SUNMOON, BST_INDETERMINATE);
    SendDlgItemMessage(hwnd, IDT_MINUTES, EM_SETREADONLY, FALSE, 0L);
    SendDlgItemMessage(hwnd, IDT_DAYS,    EM_SETREADONLY, FALSE, 0L);
    SendDlgItemMessage(hwnd, IDT_MONTHS,  EM_SETREADONLY, TRUE , 0L);
  }
  skinny_save = skinny;
  skinny = 0;
  do_datestamp( tmpStr, tmtime(gstart_time? gstart_time : time (NULL)) );
  skinny = skinny_save;
  SetDlgItemText(hwnd,IDT_DATE, tmpStr);
}

BOOL CALLBACK TextOptionsDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam) {
static int    skinny_save, marksave, graphsave;
static double marklevelsave;
int           tmpInt;


  switch(msg) {
    case WM_INITDIALOG:
      set_text_options(hwnd);
      SetDlgItemInt(hwnd,IDT_DAYS,    num_days,       FALSE);
      SetDlgItemInt(hwnd,IDT_MINUTES, increment_step, FALSE);
      SetDlgItemInt(hwnd,IDT_MONTHS,  num_months,     FALSE);
      PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, IDOK), -1);
      break;

    case WM_COMMAND:
      switch(LOWORD(wParam)) {
        case IDOK:
          tmpInt = GetDlgItemInt(hwnd,IDT_DAYS, NULL, FALSE);
          if (tmpInt > 0 && tmpInt < 100) {
             if (num_days!=tmpInt) new_params = TRUE;
             num_days = tmpInt;
          }

          tmpInt = GetDlgItemInt(hwnd,IDT_MINUTES, NULL, FALSE);
          if (tmpInt > 0 && tmpInt < 1440) {
             if (increment_step!=tmpInt) new_params = TRUE;
             increment_step = tmpInt;
          }

          tmpInt = GetDlgItemInt(hwnd,IDT_MONTHS, NULL, FALSE);
          if (tmpInt > 0 && tmpInt < 24) {
             if (num_months!=tmpInt) new_params = TRUE;
             num_months = tmpInt;
          }

          KillTimer(hwnd,1);
          skinny_save = skinny;
          if (gstart_time)
               faketime = gstart_time;
          else faketime = time( NULL );
          graphsave = graphmode;
          graphmode = 0;
          if (iscurrent) {
             marksave = mark;
             marklevelsave = marklev;
             mark    = 1;
             marklev = 0.0;
          }

          if (TextOption == ID_CALENDAR) {
            text = calendar = 1;
            skinny = 2;
            do_calendar();
          }
          else if (TextOption == ID_TIDES) {
            get_3st(hwnd, &sun_moon, IDT_SUNMOON);
            text = num_days;
            skinny = 0;
            list_tides(0);
          }
          else /* (TextOption == ID_INCREMENT)*/
            do_incremental(0);

          if (iscurrent) {
             mark = marksave;
             marklev = marklevelsave;
          }
          graphmode = graphsave;
          text = calendar = 0;
          skinny = skinny_save;
          SetTimer(hwnd,1,(UINT)TIMER_VALUE,NULL);
          EndDialog(hwnd,TRUE);
          return TRUE;

        case IDCANCEL:
          EndDialog(hwnd,FALSE);
          return TRUE;

//        case IDHELP:
//          WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Colors Menu"));
//          break;

        case ID_TIDES:
        case ID_INCREMENT:
        case ID_CALENDAR:
          TextOption = LOWORD(wParam);
          set_text_options(hwnd);
          break;

        case IDT_DATE:
          if (DateTimePicker(hwnd)) {
            notnowtime = TRUE;
            InvalidateRect(hwndMain,NULL,TRUE);
          }
          set_text_options(hwnd);
          break;

        case IDT_SUNMOON:
          if (TextOption == ID_TIDES)
             CheckDlgButton(hwnd,IDT_SUNMOON,
                IsDlgButtonChecked(hwnd,IDT_SUNMOON)? BST_UNCHECKED:BST_CHECKED);
          break;
      }
  }
  return FALSE;
}


BOOL CALLBACK CustomDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam) {
int i, tzidx, new_tz;//, CustomUseRef
static int  CustomUseRef;
static char i_custom_name[256], t_custom_name[256], ti_tzname[256], to_tzname[256];
static double ti_hmult, ti_hlevel, ti_lmult, ti_llevel;
static double to_hmult, to_hlevel, to_lmult, to_llevel;
static double ti_lat, ti_lon, ti_latc, ti_lonc, to_lat, to_lon;
static long ti_htime, ti_ltime, to_htime, to_ltime, err;
static char ti_zone[40], to_zone[40];
char t_str[256];

  switch(msg) {
    case WM_INITDIALOG:
      if (have_user_offsets) {
        strcpy(t_custom_name, custom_name);
        ti_hmult = Ihlevelmult;
        ti_hlevel= Ihtleveloff;
//        convert_level_units(&ti_hlevel, 1);
        ti_lmult = Illevelmult;
        ti_llevel= Iltleveloff;
//        convert_level_units(&ti_llevel, 1);
        ti_htime = Ihttimeoff;
        ti_ltime = Ilttimeoff;
      }
      else {
        ti_hmult = IDX_ht_mpy;
        ti_hlevel= IDX_ht_off;
        ti_lmult = IDX_lt_mpy;
        ti_llevel= IDX_lt_off;
        ti_htime = (long int)IDX_ht_time_off*60;
        ti_ltime = (long int)IDX_lt_time_off*60;
        if (!strchr("Uu", IDX_type)) {
          sprintf(t_custom_name, "Copy of %s", custom_name);
          if (TestStationExist(t_custom_name)) {
            i = 1;                      /* Find unique station name */
            do sprintf(t_custom_name, "Copy %d %s", i, custom_name);
            while (TestStationExist(t_custom_name));
          }
        }
        else
          strcpy( t_custom_name, custom_name);
      }
      strcpy(i_custom_name, t_custom_name);

      ti_lat = IDX_lat;
      ti_lon = IDX_lon;

      LoadRegionCountryState(hwnd, IDS_REGION, IDS_COUNTRY, IDS_STATE, IDS_LAT, IDS_LON,
         (float)ti_lat, (float)ti_lon, 1, IDX_zone);
      strcpy(ti_zone, build_test_ID(hwnd, IDS_REGION, IDS_COUNTRY, IDS_STATE));
      GetDlgItemText(hwnd,IDS_LAT,t_str,sizeof(t_str));
      cvt_DMM_2_deg(&ti_latc, t_str);
      GetDlgItemText(hwnd,IDS_LON,t_str,sizeof(t_str));
      cvt_DMM_2_deg(&ti_lonc, t_str);

      SetDlgItemText(hwnd,IDS_REFERENCE, IDX_reference_name);
      SetDlgItemText(hwnd,IDS_NAME,    t_custom_name);
      SetDlgItemText(hwnd,IDS_HT_TIME, makeTimeOffset(ti_htime));
      SetDlgItemText(hwnd,IDS_HT_MULT, makeRealStr(ti_hmult));
      SetDlgItemText(hwnd,IDS_HT_LEVEL,makeRealStr(ti_hlevel));
      SetDlgItemText(hwnd,IDS_LT_TIME, makeTimeOffset(ti_ltime));
      SetDlgItemText(hwnd,IDS_LT_MULT, makeRealStr(ti_lmult));
      SetDlgItemText(hwnd,IDS_LT_LEVEL,makeRealStr(ti_llevel));
      if (iscurrent) {
         SetDlgItemText(hwnd,IDS_OFF_TEXT,"Offset (kt)");
         SetDlgItemText(hwnd,IDS_MAX_TEXT,"Max &Flood");
         SetDlgItemText(hwnd,IDS_MIN_TEXT,"Max &Ebb");
      }
      else {
         SetDlgItemText(hwnd,IDS_OFF_TEXT,"Offset (ft)");
         SetDlgItemText(hwnd,IDS_MAX_TEXT,"High &tide");
         SetDlgItemText(hwnd,IDS_MIN_TEXT,"&Low tide");
      }

      CustomUseRef = !Itadjust && strlen(IDX_tzname)<=0;
      CheckDlgButton(hwnd,IDS_USEREF,CustomUseRef);
      CheckDlgButton(hwnd,IDS_USER, !CustomUseRef);

      LoadTZCombo(hwnd, IDS_TZNAME);
      SendDlgItemMessage(hwnd,IDS_TZNAME,CB_GETLBTEXT, TZComboIdx, (LPARAM)(LPSTR)&ti_tzname);

      CheckDlgButton(hwnd,IDS_UPDATE, TRUE);
      CheckDlgButton(hwnd,IDS_DISPLAY,FALSE);
      CheckDlgButton(hwnd,IDS_CONFIG, FALSE);
      CheckDlgButton(hwnd,IDS_REMOVE, BST_UNCHECKED);

      PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, IDOK), -1);
      break;

//    case WM_KEYUP:
//         if (VK_F1==wParam)
//            WinHelp(hwnd,HelpFileName,HELP_CONTEXT,IDH_CUSTOM);
//         return FALSE;

    case WM_COMMAND:
      switch(LOWORD(wParam)) {
        case IDOK:
          err = 0;

          strcpy(to_zone, build_test_ID(hwnd, IDS_REGION, IDS_COUNTRY, IDS_STATE));

          GetDlgItemText(hwnd,IDS_LAT,t_str,sizeof(t_str));
          cvt_DMM_2_deg(&to_lat, t_str);
          if ( (to_lat > 90.0) || (to_lat < -90.0) ) {
            cvt_deg_2_DMM(t_str, ti_lat);
            SetDlgItemText(hwnd,IDS_LAT, t_str);
            if (!err) err = IDS_LAT*10 + 5;
          }

          GetDlgItemText(hwnd,IDS_LON,t_str,sizeof(t_str));
          cvt_DMM_2_deg(&to_lon, t_str);
          if ( (to_lon > 180.0) || (to_lon <= -180.0)) {
            cvt_deg_2_DMM(t_str, ti_lon);
            SetDlgItemText(hwnd,IDS_LON, t_str);
            if (!err) err = IDS_LON*10 + 6;
          }

          GetDlgItemText(hwnd,IDS_HT_TIME,t_str,sizeof(t_str));
          if (verify_hhmm2sec(&to_htime, t_str)) {
            SetDlgItemText(hwnd,IDS_HT_TIME, makeTimeOffset(ti_htime));
            if (!err) err = IDS_HT_TIME*10 + 1;
          }

          GetDlgItemText(hwnd,IDS_LT_TIME,t_str,sizeof(t_str));
          if (verify_hhmm2sec(&to_ltime, t_str)) {
            SetDlgItemText(hwnd,IDS_LT_TIME, makeTimeOffset(ti_ltime));
            if (!err) err = IDS_LT_TIME*10 + 1;
          }

          if (!GetDlgItemDouble(hwnd, IDS_HT_MULT, &to_hmult) ||
             (to_hmult > 5.0) || (to_hmult < 0.2)) {
            SetDlgItemText(hwnd,IDS_HT_MULT, makeRealStr(ti_hmult));
            if (!err) err = IDS_HT_MULT*10 + 2;
          }

          if (!GetDlgItemDouble(hwnd, IDS_LT_MULT, &to_lmult) ||
             (to_lmult > 5.0) || (to_lmult < 0.2)) {
            SetDlgItemText(hwnd,IDS_LT_MULT, makeRealStr(ti_lmult));
            if (!err) err = IDS_LT_MULT*10 + 2;
          }

          if (!GetDlgItemDouble(hwnd, IDS_HT_LEVEL, &to_hlevel) ||
             (to_hlevel > 10.0) || (to_hlevel < -10.0)) {
            SetDlgItemText(hwnd,IDS_HT_LEVEL,makeRealStr(ti_hlevel));
            if (!err) err = IDS_HT_LEVEL*10 + 3;
          }

          if (!GetDlgItemDouble(hwnd, IDS_LT_LEVEL, &to_llevel) ||
             (to_llevel > 10.0) || (to_llevel < -10.0)) {
            SetDlgItemText(hwnd,IDS_LT_LEVEL,makeRealStr(ti_llevel));
            if (!err) err = IDS_LT_LEVEL*10 + 3;
          }

          GetDlgItemText(hwnd,IDS_NAME,t_custom_name,sizeof(t_custom_name));
          if (!err && !strlen(t_custom_name) &&
              ((to_hmult != 1.0) || (to_hlevel != 0.0) ||
               (to_lmult != 1.0) || (to_llevel != 0.0) ||
               (to_htime != 0)   || (to_ltime  != 0)))
             err = IDS_NAME*10 + 4;

//          new_tz = GetTZCombo(hwnd, IDS_TZNAME, IDS_USER, 0);
          GetWindowText(GetDlgItem(hwnd, IDS_TZNAME), (LPSTR)&to_tzname, sizeof(to_tzname));
          if (!isalpha(to_tzname[0])
             && BST_CHECKED == IsDlgButtonChecked(hwnd,IDS_UPDATE)
             && BST_CHECKED != IsDlgButtonChecked(hwnd,IDS_REMOVE))
             if (!err) err = IDS_TZNAME*10 + 7;
          new_tz = strcmp(ti_tzname, to_tzname);

          if (stricmp(t_custom_name, i_custom_name) &&
              tolower(TestStationExist(t_custom_name))=='u')
             if (!err) err = IDS_NAME*10 + 8;

          if (err) {
             char *msgs[]={
               {"Time must be in the form +/-HH:MM"},                         /* 1 */
               {"Multipliers must be in the range 0.2 to 5.0"},               /* 2 */
               {"Level offsets must be in the range -10.0 to 10.0"},          /* 3 */
               {"Stations must have a name"},                                 /* 4 */
               {"Latitude must be between -90.0 and +90.0"},                  /* 5 */
               {"Longitude must be between -180.0 and +180.0"},               /* 6 */
               {"User station file entries can only have standard timezones"},/* 7 */
               {"A user station with that name already exists"}               /* 8 */
             };
             MessageBox(hwnd, msgs[(err%10)-1], "Field Error", MB_ICONEXCLAMATION);
             PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, err/10), -1);
             return FALSE;
          }

          if (FSAME(to_hmult, ti_hmult) && FSAME(to_hlevel, ti_hlevel) &&
              FSAME(to_lmult, ti_lmult) && FSAME(to_llevel, ti_llevel) &&
              FSAME(to_htime, ti_htime) && FSAME(to_ltime,  ti_ltime ) &&
              FSAME(to_lat,   ti_latc ) && FSAME(to_lon,    ti_lonc  ) &&
                 !strcmp(ti_zone, to_zone) &&  !new_tz &&
                 !strcmp(t_custom_name, i_custom_name) &&
                 BST_CHECKED != IsDlgButtonChecked(hwnd,IDS_REMOVE) &&
              ( ( have_user_offsets && new_tz && !have_new_custom)||
                 !have_user_offsets ) ) {
             MessageBox(hwnd, "No settings changed, ignoring",
                   "No custom changes", MB_ICONEXCLAMATION);
             EndDialog(hwnd,FALSE);
             return FALSE;
          }
// We have committed so update the variables

          Ihlevelmult = to_hmult;
//          convert_level_units(&to_hlevel, 0);
          Ihtleveloff = to_hlevel;

          Illevelmult = to_lmult;
//          convert_level_units(&to_llevel, 0);
          Iltleveloff = to_llevel;

          Ihttimeoff = to_htime;
          Ilttimeoff = to_ltime;

          Ilat = to_lat;
          Ilon = to_lon;
          strcpy(Izone, to_zone);

          if     (!iscurrent &&  strstr(t_custom_name, "Current"))
            *strstr(t_custom_name, "Current") = 'c';
          else if (iscurrent && !strstr(t_custom_name, "Current")) {
            if (strlen(t_custom_name) < (MAXNAMELEN-strlen("Current")))
                 strcat(t_custom_name, "Current");
            else strcpy(t_custom_name + (MAXNAMELEN-strlen("Current")), "Current");
          }

          strADupe(&custom_name, t_custom_name);
          WinTideCheckMenus(hwndMain);

// Function: Generate config file
          if (BST_CHECKED == IsDlgButtonChecked(hwnd,IDS_CONFIG)) {
            have_user_offsets = TRUE;
            have_new_custom   = TRUE;
            new_params = TRUE;
            GetTZCombo(hwnd, IDS_TZNAME, IDS_USER, 0);    // Enable force set tzname
            for (i=0; i<strlen(custom_name) && i<64; i++) // Make name suitable for filename
               if (!isalnum(custom_name[i]) && !strchr(" _",custom_name[i]))
                   custom_name[i] = ' ';
            custom_name[i] = '\0';
            strcpy(location, IDX_reference_name);
            load_location_data( location, 0 );
            NameWindow(hwndMain, custom_name);
            EndDialog(hwnd,TRUE);
            break;
          }
// Function: Do not save, just display
          else if (BST_CHECKED == IsDlgButtonChecked(hwnd,IDS_DISPLAY)) {
            have_user_offsets = TRUE;
            have_new_custom   = TRUE;
            new_params = TRUE;
            GetTZCombo(hwnd, IDS_TZNAME, IDS_USER, 0);  // Enable force set tzname
            strcpy(location, IDX_reference_name);
            load_location_data( location, 0 );
            NameWindow(hwndMain, custom_name);
          }
// Function: Remove from user file
          else if ((BST_CHECKED == IsDlgButtonChecked(hwnd,IDS_UPDATE))
                && (BST_CHECKED == IsDlgButtonChecked(hwnd,IDS_REMOVE)) ) {
            if (tolower(TestStationExist(custom_name))!='u') {
              MessageBox(hwnd, "That station is not in the user station list",
                               "Delete user station", MB_ICONEXCLAMATION|MB_OK);
              EndDialog(hwnd,FALSE);
              break;
            }
            else {
              if (IDOK!=MessageBox(hwnd, "Are you sure you want to delete this station?",
                                "Delete user station", MB_ICONEXCLAMATION|MB_OKCANCEL)) {
                EndDialog(hwnd,FALSE);
                break;
              }
              UserStationFuncs(USF_REMOVE, custom_name);
              remove_mru(custom_name);
              Ihttimeoff = Ilttimeoff = 0;
              Ihlevelmult= Illevelmult= 1.0;
              Ihtleveloff= Iltleveloff= 0.0;
              have_user_offsets = FALSE;
              have_new_custom   = FALSE;
              new_params = TRUE;
              strADupe(&custom_name, IDX_reference_name);
              WinTideCheckMenus(hwndMain);
              strcpy(location, IDX_reference_name);
              load_location_data( location, 0 );
              NameWindow(hwndMain, custom_name);
            }
          }
// Function: Add/Update from user file
          else if (BST_CHECKED == IsDlgButtonChecked(hwnd,IDS_UPDATE)) {
            have_user_offsets = FALSE;
            have_new_custom   = FALSE;
            new_params = TRUE;
            strcpy(tadjust_last, to_tzname);
            UserStationFuncs(USF_UPDATE, custom_name);
            strcpy(location, custom_name);
            load_location_data( location, 0 );
            NameWindow(hwndMain, custom_name);
          }
          EndDialog(hwnd,FALSE);
          break;

        case IDS_USEREF:
          tzidx=SendDlgItemMessage(hwnd,IDS_TZNAME,CB_FINDSTRING,-1,(LPARAM)((LPSTR)(tzfile+1)));
          if (tzidx==CB_ERR) tzidx=0;
          SendDlgItemMessage(hwnd,IDS_TZNAME, CB_SETCURSEL, tzidx, 0L);
          CheckDlgButton(hwnd,IDS_USEREF,CustomUseRef);
          CheckDlgButton(hwnd,IDS_USER, !CustomUseRef);
          break;

        case IDS_TZNAME:
          if ((HIWORD(wParam)) == CBN_SELCHANGE) {
             CustomUseRef = FALSE;
             CheckDlgButton(hwnd,IDS_USEREF,CustomUseRef);
             CheckDlgButton(hwnd,IDS_USER, !CustomUseRef);
          }
          break;

        case IDS_UPDATE:
          if  ((BST_UNCHECKED == IsDlgButtonChecked(hwnd,IDS_UPDATE))
            || (tolower(TestStationExist(custom_name))!='u')          )
            CheckDlgButton(hwnd,IDS_REMOVE, BST_UNCHECKED);
          break;

        case IDS_DISPLAY:
        case IDS_CONFIG:
          CheckDlgButton(hwnd,IDS_REMOVE, BST_UNCHECKED);
          break;

        case IDS_REMOVE:
          if    ((BST_CHECKED   == IsDlgButtonChecked(hwnd,IDS_UPDATE))
              && (BST_UNCHECKED == IsDlgButtonChecked(hwnd,IDS_REMOVE))
              && (tolower(TestStationExist(custom_name))=='u')          )
            CheckDlgButton(hwnd,IDS_REMOVE, BST_CHECKED);
          else
            CheckDlgButton(hwnd,IDS_REMOVE, BST_UNCHECKED);
          break;

        case IDCANCEL:
          EndDialog(hwnd,FALSE);
          break;

        case IDHELP:
          WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Custom Station Menu"));
          break;
      }
  }
  return FALSE;
}

/*-----------------5/29/2005 8:17PM-----------------
 * DrawTrayIcon - Draw animated tide icon in tooltray
 * --------------------------------------------------*/
HICON DrawTrayIcon(HWND hwnd) {
#define WXT_ICON_CLRS     2  // Number of colors used in icon image
#define WXT_ICON_IMG_BPP  4  // Image Bits Per Pixel
#define WXT_ICON_MSK_BPP  1  // Mask Bits Per Pixel

HDC              hDC;
BITMAPINFO       *bmImg,  *bmMsk;
BITMAPINFOHEADER *biImg,  *biMsk;
RGBQUAD          *rgbImg, *rgbMsk, tc;
LPSTR            pxImg,   pxMsk, lp;
ICONINFO         ici;
HICON            newIcon=NULL;
static HBITMAP   bm1=NULL, bc1=NULL;    // Reused compatible bitmap buffers
int              icon_size= GetSystemMetrics(SM_CXICON);
int              imgLenLn = (int)(icon_size * WXT_ICON_IMG_BPP + 3) / 8;
int              imgBytes = imgLenLn * icon_size;
int              mskLenLn = (int)(icon_size * WXT_ICON_MSK_BPP + 3) / 8;
int              mskBytes = mskLenLn * icon_size;
int              rtn=0, x, y, i, llx, lox, iconBorder, iconLines;
time_t           this_time;

  // Allocate memory for color image and monochrome mask bitmaps
  if (NULL==(bmImg =malloc(sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD)*WXT_ICON_CLRS +imgBytes))) return NULL;
  if (NULL==(bmMsk =malloc(sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD)*WXT_ICON_CLRS +mskBytes))) {
    free(bmImg);
    return NULL;
  }
  biImg = (BITMAPINFOHEADER *)bmImg;
  biMsk = (BITMAPINFOHEADER *)bmMsk;

  // Set bitmap descriptors for color image and monochrome mask
  biImg->biSize         = biMsk->biSize         = sizeof(BITMAPINFOHEADER);
  biImg->biWidth        = biMsk->biWidth        = icon_size;
  biImg->biHeight       = biMsk->biHeight       = icon_size;
  biImg->biPlanes       = biMsk->biPlanes       = 1;
  biImg->biBitCount                             = WXT_ICON_IMG_BPP;
  biImg->biCompression  = biMsk->biCompression  = BI_RGB;
  biImg->biXPelsPerMeter= biMsk->biXPelsPerMeter= 0;
  biImg->biYPelsPerMeter= biMsk->biYPelsPerMeter= 0;
  biImg->biClrUsed                              = WXT_ICON_CLRS;
  biImg->biClrImportant                         = WXT_ICON_CLRS;
  biImg->biSizeImage                            = sizeof(BITMAPINFOHEADER)         +
                                                  sizeof(RGBQUAD)*biImg->biClrUsed +
                                                  imgBytes;
  // .. then set monochrome mask unique values
  biMsk->biBitCount     = WXT_ICON_MSK_BPP;
  biMsk->biClrUsed      = 0;
  biMsk->biClrImportant = 0;
  biMsk->biSizeImage    = sizeof(BITMAPINFOHEADER)         +
                          sizeof(RGBQUAD)*biMsk->biClrUsed +
                          mskBytes;

  // Setup pointers to different sections in bitmap descriptors
  rgbImg= (RGBQUAD *)((LPSTR)bmImg + sizeof(BITMAPINFOHEADER));
  rgbMsk= (RGBQUAD *)((LPSTR)bmMsk + sizeof(BITMAPINFOHEADER));
  pxImg = (LPSTR)bmImg + sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD)*biImg->biClrUsed;
  pxMsk = (LPSTR)bmMsk + sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD)*biMsk->biClrUsed;

  // Clear RGB color blocks and image pixel data areas
  memset(rgbImg, 0, biImg->biSizeImage - sizeof(BITMAPINFOHEADER));
  memset(rgbMsk, 0, biMsk->biSizeImage - sizeof(BITMAPINFOHEADER));

#define SetBmpClr(n,c) {tc.rgbRed=GetRValue(c);tc.rgbGreen=GetGValue(c);tc.rgbBlue=GetBValue(c);rgbImg[n]=tc;}

  // Select rising or falling color to fill icon
  this_time = time (NULL);
  rising    = -1;
  watch_tides (this_time);
  if (iscurrent)
    rising   = this_tide > (-fakedatum / fakeamplitude);
  SetBmpClr(0,rising? fgrise_color : fgfall_color);

  // Draw Icon
  if      (IconStyle==1) {              // 1 = Small Arrow
    iconBorder = 0;
    iconLines  = 8;
    y = (int)(where_in_current_tide(this_time) * (icon_size-iconLines));
  }
  else {                                // All others have line at level
    y = rising? 0 : icon_size-1;
    if      (IconStyle==2) {            // 2 = Arrow + Line
      for (i=0; i<icon_size-2 && y >= 0 && y < icon_size; i++) {
        lp = pxImg + imgLenLn * (icon_size-1 - y) + icon_size/4; // Center of line y
        if (i<icon_size/2)
             x = i/3;
        else x = icon_size/12;
        *(lp-x-1) = 0x11;
        *(lp+x  ) = 0x11;
        y += rising? 1 : -1;
      }
    }
    else if (IconStyle==3) {            // 3 = Diagonal + Line
      for (i=0; i<icon_size && y >= 0 && y < icon_size; i++) {
        lp = pxImg + imgLenLn * (icon_size-1 - y);
        x  = icon_size/2 - i/2 - 1;
        *(lp+x) = 0x11;
        y += rising? 1 : -1;
      }
    }
    // Draw line at tide level relative to total tide range
    iconBorder = 2;
    iconLines  = 2;
    y = (int)(where_in_current_tide(this_time) * (icon_size-iconBorder*2)) + iconBorder;
  }

  y += rising? iconLines : -iconLines;
  if      (y > (icon_size-1-iconBorder)) y = icon_size-1-iconBorder;
  else if (y <            0+iconBorder ) y = 0+iconBorder;

  // Draw horizontal line 2 lines thick in case icon is reduced size
  SetBmpClr(1,fgmllw_color);
  for (i=0; i<iconLines && y >= 0 && y < icon_size; i++) {
    lox = i*2;
    llx = icon_size - lox*2;
    lp = pxImg + imgLenLn * (icon_size-1 - y) + lox/2;
    for (x=0; x<llx/2; x++) *lp++ = 0x11;
    y += rising? -1 : 1;
  }

  // Now create the device-dependent bitmaps used to build the icon
  hDC = GetDC(hwnd);
  if (bc1==NULL) bc1 = CreateCompatibleBitmap(hDC, biImg->biWidth, biImg->biHeight);
  if (bm1==NULL) bm1 = CreateCompatibleBitmap(hDC, biMsk->biWidth, biMsk->biHeight);

  if (bc1 && bm1) {
    // Set the DIBs to DDBs
    rtn = (32==SetDIBits(hDC, bc1, 0, biImg->biHeight, pxImg, bmImg, DIB_RGB_COLORS));
    rtn&= (32==SetDIBits(hDC, bm1, 0, biMsk->biHeight, pxMsk, bmMsk, DIB_RGB_COLORS));
  }
  ReleaseDC(hwnd, hDC);
  free(bmImg);
  free(bmMsk);

  // Update icon handle in tray descriptor (but let caller notify of change)
  if (rtn) {
    ici.fIcon    = TRUE;
    ici.xHotspot = ici.yHotspot = 0;
    ici.hbmMask  = bm1;
    ici.hbmColor = bc1;
    newIcon = CreateIconIndirect(&ici);
  }
//  DeleteObject(bm1); // Compatible bitmap buffers are reused to reduce memory leak
//  DeleteObject(bc1);
  return (newIcon);
}

/*-----------------5/30/2005 8:41AM-----------------
 *
 * --------------------------------------------------*/
void DoTrayIcon(HWND hwnd, int new_minute, time_t now_time) {
  HICON newIcon;
  if (UseIcon == 2 || (IsIconic(hwnd) && UseIcon > 0))  {
    if ( !inTray || new_minute || ForceUpdateIcon ) {
      /*  We don't have an icon in the system tray or we have to update */
      trayIcon.cbSize   = sizeof(NOTIFYICONDATA);
      trayIcon.hWnd     = hwnd;
      trayIcon.uID      = 1;
      trayIcon.uFlags   = NIF_MESSAGE | NIF_ICON | NIF_TIP;
      trayIcon.uCallbackMessage = ID_TRAY_CLICK;
//      trayIcon.hIcon    = trayicon;
      if (NULL!=(newIcon=DrawTrayIcon(hwnd))) {
        char *sType;
        if (trayicon != NULL)
          DestroyIcon(trayicon);
        trayicon       = newIcon;
        trayIcon.hIcon = trayicon;
        /* Add intro with upcoming event and time remaining to that event.
           This relies on event_type and next_ht_adj left over from DrawTrayIcon */
        if      (event_type & 1 &&  iscurrent) sType = "Max ebb";
        else if (event_type & 1 && !iscurrent) sType = "Low tide";
        else if (event_type & 2 &&  iscurrent) sType = "Max flood";
        else /* (event_type & 2 && !iscurrent*/sType = "High tide";
        sprintf(trayIcon.szTip, "%s to %s at ", 1+seconds2hhmm (next_ht_adj-now_time), sType);
        strncpy(trayIcon.szTip+strlen(trayIcon.szTip), custom_name, 63-strlen(trayIcon.szTip));
        trayIcon.szTip[63]= '\0';
      }
      else {
        /*  We couldn't build icon so use default */
        trayIcon.hIcon = wxtide32icon;
        strncpy(trayIcon.szTip, custom_name, 63);
        trayIcon.szTip[63]= 0x00;
      }


      if (inTray)
           inTray = Shell_NotifyIcon(NIM_MODIFY, &trayIcon) != 0;
      else inTray = Shell_NotifyIcon(NIM_ADD,    &trayIcon) != 0;
      /* We just updated the icon.  If we did not get a handle back, then delete the old icon and try again.
         This seems to happen only occasionally after restart from Hibernate/Standby mode...
         and yes, I pulled lots of hair finding the problem and how to get around it! */
      if (!inTray) {                     // Something bad happened, try again
        Shell_NotifyIcon(NIM_DELETE, &trayIcon);
        inTray = Shell_NotifyIcon(NIM_ADD, &trayIcon) != 0;
      }
    }
  }
  else if (inTray) {
    /*  We have an icon in the system tray but we shoudn't, so remove it */
    ShowWindow(hwnd, SW_SHOW);
    Shell_NotifyIcon(NIM_DELETE, &trayIcon);
    inTray = FALSE;
    if (trayicon != NULL) {
      DestroyIcon(trayicon);
      trayicon = NULL;
    }
  }
  if (inTray && IsIconic(hwnd) && UseIcon > 0)
    /*  We have a tray icon and hwndMain is minimized, so hide main window task bar entry */
    ShowWindow(hwnd, SW_HIDE);
  ForceUpdateIcon   = 0;
}

void DrawDC(HDC hdc) {
HCURSOR hcurSave;
hcurSave = SetCursor(LoadCursor(NULL, IDC_WAIT));
  if (!IsIconic(hwndMain) && !hold_restore) {
    display = hdc;
    if (gfont == NULL) {
      LOGFONT lf;
      memset(&lf, 0, sizeof(lf));
      lf.lfHeight = -MulDiv(gFontSize, GetDeviceCaps(hdc, LOGPIXELSY), 72);
      strcpy(lf.lfFaceName,gFontName);
      gfont=CreateFontIndirect(&lf);
      textH = -lf.lfHeight;
    }
    SelectFont(hdc,gfont);
    SetTextColor(hdc,fgtext_color);
    SetBkMode(hdc,TRANSPARENT);
    /* Draw_graph does not return. */
    faketime = gstart_time;
    if (graphmode) {
       /* See above. */
       if (!faketime) {
          faketime = time (NULL);
          if (now)
            faketime -= tstep * (winW>>1);
       }
       draw_graph ();
    }
    else {

    /* This clumsy calibration step is necessary by using arbitrary offsets in clock mode. */
       time_t this_time;
       if (testmode)
            this_time = faketime;
       else this_time = time(NULL);
       win_assert (this_time > DAYSECONDS*1.5);
       next_ht = (time_t)(this_time - DAYSECONDS*1.5);
       prev_ht_adj = next_ht_adj = 0;
       while (this_time >= prev_ht_adj + 70)
          update_high_tide ();
       if (hairy) set_hairy_water_level ();
       else       set_water_level ();
    }
    DoCrosshair(hdc);
    SetCursor(hcurSave);
  }
}

// ********************************************************************** //
// ** This file was downloaded from Jeff Heaton's Source Code Examples ** //
// **      www.heat-on.com/source  or  www.jeffheaton.com/source       ** //
// ********************************************************************** //
// Copyright 1999 by Jeff Heaton
// ********************************************************************** //
// Permission is granted for royalty-free use in compiled form.  No
// additional licensing is required.  This code may not be distributed
// in source form without written premission by Jeff Heaton.  Generally
// I will grant it.  Just ask first.
//
// <Edit> Authorization to include in WXTide32 received 5/30/2005
//
// Basically I do not care how you distribute this code in compiled form.
// But if your going to publish my source code somewhere I'd like to hear
// about it first.
// ********************************************************************** //

// SnapBMP V1.0
// Written 1/14/1999 by Jeff Heaton
// Modified by Mike Hopper to only take WXTide32 graphics window
//
// Capture the screen in any mode and save as a .bmp.
//
// This is a COM object that allows programs to capture the screen to a bitmap file.
// One single method is provided, Snap, which actually does this process.

// Function Name: SaveBitmap
// Parameters:
//    name   - Filename to save the bitmap to
//    bitmap - Bitmap handle to save
// Returns:
//    TRUE if successful.
// Purpose:
//    This is used to save a HBITMAP handle as a Windows .bmp file.
// Author
//    Jeff Heaton

BOOL SaveBitmap(char *name,HBITMAP bitmap) {
  HDC  hdcScreen;          // Device context of the screen
  HWND hwndScreen;         // Desktop window handle
  int  i;                  // Counts the number of scanlines
  BYTE *bits;              // Pointer to the bitmap's bits
  BITMAPINFOHEADER *info;  // Info structure about this bitmap
  HANDLE fp;               // A file handle
  BITMAPFILEHEADER finfo;  // File info header
  long biHeight;           // Various sizes needed to build the file
  long ctsize,imgsize,pixels;// Various sizes needed to build the file
  DWORD dw;                // Doubleword return code

  hwndScreen= GetDesktopWindow();
  hdcScreen = GetDC( hwndScreen );

  info = (BITMAPINFOHEADER*)GlobalAlloc(GMEM_FIXED,sizeof(BITMAPFILEHEADER)+(1024*sizeof(RGBQUAD)));
  info->biSize     = sizeof(BITMAPINFOHEADER);
  info->biBitCount = 0;

  i = GetDIBits( hdcScreen,// handle of device context
      bitmap,              // handle of bitmap
      0,                   // first scan line to set in destination bitmap
      0,                   // number of scan lines to copy
      NULL,                // address of array for bitmap bits
      (BITMAPINFO*)info,   // address of structure with bitmap data
      DIB_RGB_COLORS       // RGB or palette index
      );

  if( (info->biBitCount!=4) &&
      (info->biBitCount!=8) &&
      (info->biBitCount!=24)  )
    info->biBitCount = 24;

  info->biCompression = BI_RGB;

  biHeight = (long)fabs(info->biHeight);
  pixels   = info->biWidth * biHeight;

  switch(info->biBitCount)
  {  /* BITMAPINFO says lines have to pad to DWORD boundries */
    case  4:ctsize=16; imgsize=pixels/2 +3*biHeight;break;
    case  8:ctsize=256;imgsize=pixels   +3*biHeight;break;
    case 16:ctsize=0;  imgsize=pixels*2 +2*biHeight;break;
    case 24:ctsize=0;  imgsize=pixels*3 +3*biHeight;break;
    case 32:ctsize=0;  imgsize=pixels*4            ;break;
    default:
      ReleaseDC( hwndScreen,hdcScreen );
      return FALSE;
  }

  bits = (unsigned char*)GlobalAlloc(GMEM_FIXED,imgsize);

  i = GetDIBits( hdcScreen,// handle of device context
      bitmap,              // handle of bitmap
      0,                   // first scan line to set in destination bitmap
      biHeight,            // number of scan lines to copy
      bits,                // address of array for bitmap bits
      (BITMAPINFO*)info,   // address of structure with bitmap data
      DIB_RGB_COLORS       // RGB or palette index
      );

  finfo.bfType = 0x4D42;   // Byte swapped WORD equiv to "BM"

  finfo.bfOffBits=sizeof(BITMAPFILEHEADER)+
                  sizeof(BITMAPINFOHEADER)+
                 (sizeof(RGBQUAD)*ctsize);
  finfo.bfSize = finfo.bfOffBits+imgsize;
  finfo.bfReserved1 = 0;
  finfo.bfReserved2 = 0;

  fp = CreateFile(name,    // pointer to name of the file
       GENERIC_WRITE,      // access (read-write) mode
       0,                  // share mode
       NULL,               // pointer to security attributes
       CREATE_ALWAYS,      // how to create
       FILE_ATTRIBUTE_NORMAL|FILE_FLAG_SEQUENTIAL_SCAN,// file attributes
       NULL                // handle to file with attributes to copy
       );

  if( fp==INVALID_HANDLE_VALUE ) {
    GlobalFree(info);
    GlobalFree(bits);
    ReleaseDC( hwndScreen,hdcScreen );
    return FALSE;
  }

  WriteFile(fp,&finfo,sizeof(BITMAPFILEHEADER),&dw,NULL);
  WriteFile(fp,  info,sizeof(BITMAPINFOHEADER)+sizeof(RGBQUAD)*ctsize,&dw,NULL);
  WriteFile(fp,  bits,imgsize,&dw,NULL);
  CloseHandle(fp);
  GlobalFree(info);
  GlobalFree(bits);

  ReleaseDC( hwndScreen,hdcScreen );
  return TRUE;
}

// Function Name: CaptureWindow (was ScreenCapture)
// Parameters:
//    Handle of window to copy
// Returns:
//    A bitmap handle, or NULL on failure.
// Purpose:
//    This is used to capture a window as a bitmap.
// Author
//    Jeff Heaton

HBITMAP CaptureWindow(HWND hwndWindow, int width, int heigth) {
  HDC     hdcWindow;     // Device context of the screen
  HBITMAP captureBmp;    // Bitmap to capture into
  HBITMAP oldBmp;        // Previously selected bitmap in the "capture context"
  HDC     hdcCapture;    // Device context to select the capture bitmap into
  BOOL    b;             // Misc error flag

  hdcWindow = GetDC(hwndWindow);

  if( (captureBmp = CreateCompatibleBitmap( hdcWindow, width, heigth )) == NULL )
  {
    ReleaseDC( hwndWindow,hdcWindow );
//    ErrorHandler("Could not create a compatible bitmap.");
    return NULL;
  }
  if( (hdcCapture = CreateCompatibleDC( hdcWindow )) == NULL )
  {
    DeleteObject(captureBmp);
    ReleaseDC( hwndWindow,hdcWindow );
//    ErrorHandler("Could not create a device context.");
    return NULL;
  }

  oldBmp = (HBITMAP)SelectObject( hdcCapture, captureBmp );

  b = BitBlt( hdcCapture,// handle to destination device context
              0,         // x-coord of destination rectangle upper-left corner
              0,         // y-coord of destination rectangle upper-left corner
              width,     // width of destination rectangle
              heigth,    // height of destination rectangle
              hdcWindow, // handle to source device context
              0,         // x-coord of source rectangle upper-left corner
              0,         // y-coord of source rectangle upper-left corner
              SRCCOPY    // raster operation code
              );

  if( !b ) {
    SelectObject( hdcCapture,oldBmp );
    DeleteDC( hdcCapture );
    DeleteObject(captureBmp );
    ReleaseDC( hwndWindow,hdcWindow );
//    ErrorHandler("Could not create a device context.");
    return NULL;
  }

  // Clean up

  SelectObject( hdcCapture,oldBmp );
  DeleteDC( hdcCapture );
  ReleaseDC( hwndWindow,hdcWindow );
  return captureBmp;
}

/* LCC is missing these defines */
//#ifndef PBT_APMQUERYSUSPEND
//#define PBT_APMQUERYSUSPEND             0x0000
//#define PBT_APMQUERYSTANDBY             0x0001
//#define PBT_APMQUERYSUSPENDFAILED       0x0002
//#define PBT_APMQUERYSTANDBYFAILED       0x0003
//#define PBT_APMSUSPEND                  0x0004
//#define PBT_APMSTANDBY                  0x0005
//#define PBT_APMRESUMECRITICAL           0x0006
//#define PBT_APMRESUMESUSPEND            0x0007
//#define PBT_APMRESUMESTANDBY            0x0008
//#define PBT_APMBATTERYLOW               0x0009
//#define PBT_APMPOWERSTATUSCHANGE        0x000A
//#define PBT_APMOEMEVENT                 0x000B
//#define PBT_APMRESUMEAUTOMATIC          0x0012
//#endif

LRESULT CALLBACK TidesWndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam) {
PAINTSTRUCT ps;
static BOOL print_graph = 0, ctl_key = 0;
char szMessage[256];

  switch(msg) {
    case WM_CREATE:
      if (!have_index) {
//         init_index_file(TRUE, hwndMain);    // Load full info database
//         load_location_info( location, IDX_rec_num );
         load_location_data( location, IDX_rec_num );
      }
      WinTideCheckMenus(hwnd);
      SetTimer(hwnd,1,(UINT)TIMER_VALUE,NULL);
      return FALSE;

/*-----------------12/16/2005 11:25AM---------------
 * Both the window size and window move routines are extremely fragile.
 * Trying to handle all cases is difficult, hence assorted interlocks.
 * --------------------------------------------------*/
    case WM_SIZE:
      if (!hold_main_window && save_windows && !hold_size_min_max && !hold_restore &&
          (int)wParam!=SIZE_MINIMIZED &&
          (int)wParam!=SIZE_MAXIMIZED   ) {
         new_params = TRUE;
         WinTideCheckMenus(hwnd);
      }
      hold_size_min_max = (int)wParam==SIZE_MINIMIZED || (int)wParam==SIZE_MAXIMIZED;
      winW=LOWORD(lParam) < 50? 50 : LOWORD(lParam);
      winH=HIWORD(lParam) < 50? 50 : HIWORD(lParam);
      InvalidateRect(hwnd,NULL,TRUE);
      GetWindowRect(hwnd,&WinDimMain);
      DoScrollBar(graphmode&&usescrollbar);
      if (!hold_restore && !hold_main_window) section_save();
      if (wParam == SIZE_MINIMIZED)
        PostMessage(hwnd, WM_PAINT, 0, 0);
      else
        WindowWasMax = (int)wParam==SIZE_MAXIMIZED;
      ForceUpdateIcon = 1;
      return FALSE;

    case WM_MOVE:
      winX=(int)(short int)LOWORD(lParam);
      winY=(int)(short int)HIWORD(lParam);
      GetWindowRect(hwnd,&WinDimMain);
      if (!hold_restore && !hold_main_window) section_save();
      hold_move_min_max = IsIconic(hwnd) || IsZoomed(hwnd);
      if (!hold_main_window && save_windows && !hold_size_min_max && !hold_restore && !hold_move_min_max) {
        new_params = TRUE;
        WinTideCheckMenus(hwnd);
      }
      return FALSE;

/* Timer is called every second, updates hairy and clock display each system minute change */
    case WM_TIMER:
      {
        static int min = -1, bells = 0;
        time_t     tod = time(NULL);
        struct tm *t   = localtime( &tod );
        int new_minute = min != t->tm_min;
        min = t->tm_min;
        if (new_minute && (t->tm_min == 0 || t->tm_min == 30) && haveSounds && ShipBells) {
          SYSTEMTIME st;
          GetLocalTime(&st);
          bells = (st.wHour % 4)*2;   // 2 bells each hour
          if (t->tm_min) bells++;       // 1 bell on 30 minutes
          if (!bells)    bells = 8;     // 0 is 8 bells
          bellsname[0] = bells + '0';
          PlaySound(bellsname, NULL, SND_ASYNC | SND_FILENAME);
        }
        if (new_minute && (hairy || !(graphmode || overview))) {
          InvalidateRect(hwnd,NULL,TRUE);
          UpdateWindow(hwnd);
        }
        if (new_minute || ForceUpdateIcon)
          DoTrayIcon(hwnd, new_minute, tod);
        if (auto_save && new_params && !hold_main_window && /*!hold_size_min_max &&*/ !hold_restore &&
            !cant_open_config_file) {
          KillTimer(hwnd,1);
          write_config_file(szConfigPath, TRUE);
          WinTideCheckMenus(hwnd);
          SetTimer(hwnd,1,(UINT)TIMER_VALUE,NULL);
        }
        return FALSE;
      }

//    case WM_POWERBROADCAST:
//      if ((int)wParam == PBT_APMSUSPEND)
//        KillTimer(hwnd,1);
//      else if ((int)wParam == PBT_APMRESUMECRITICAL  ||
//               (int)wParam == PBT_APMRESUMESUSPEND   ||
//               (int)wParam == PBT_APMRESUMEAUTOMATIC ||
//               (int)wParam == PBT_APMRESUMESTANDBY     )
//        SetTimer(hwnd,1,(UINT)TIMER_VALUE,NULL);
//      return FALSE;

    case WM_PAINT:
      BeginPaint(hwnd,&ps);
      DrawDC(ps.hdc);
      EndPaint(hwnd,&ps);
      return FALSE;

    case WM_CLOSE:
      if (new_params) {
        int ans=IDYES;
        if (auto_save ||
          (IDYES==(ans=MessageBox(hwnd,"Parameters have changed.\r\n"
                                       "Save before exit?",
             "Save parameters",MB_ICONEXCLAMATION|MB_YESNOCANCEL))))
          write_config_file(szConfigPath, TRUE);
        if (ans == IDCANCEL)
          return TRUE;
      }
      KillTimer(hwnd,1);
      if (inTray)
         Shell_NotifyIcon(NIM_DELETE, &trayIcon);
      PostQuitMessage(0);
      return TRUE;

    case WM_CONTEXTMENU:
      TrackPopupMenu( GetSubMenu(GetMenu(hwndMain),0), TPM_RIGHTBUTTON |TPM_TOPALIGN |TPM_LEFTALIGN,
            LOWORD(lParam), HIWORD(lParam), 0, hwnd, NULL);
      return FALSE;

    case ID_TRAY_CLICK:
      if (lParam == WM_LBUTTONUP) {
        if (!inTray) { // If we got a click but no icon, Hibernate error!  Delete icon to force redraw */
          Shell_NotifyIcon(NIM_DELETE, &trayIcon);
          if (trayicon != NULL) {
            DestroyIcon(trayicon);
            trayicon = NULL;
          }
        }
        PostMessage (hwnd, WM_COMMAND, ID_SHOW_HIDE, 0);
      }
      else if (lParam == WM_RBUTTONUP) {
        HMENU iconMenu, subMenu;
        POINT here;
        GetCursorPos(&here);
        SetForegroundWindow(hwnd);
        subMenu  = GetSubMenu(GetMenu(hwndMain),0);
        iconMenu = CreatePopupMenu();
        AppendMenu(iconMenu, MF_STRING, ID_SHOW_HIDE, (IsIconic(hwnd)? "Show" : "Hide"));
        SetMenuDefaultItem( iconMenu, 0, 1);
        AppendMenu(iconMenu, MF_SEPARATOR , 0, 0);
        AppendMenu(iconMenu, MF_STRING, ID_ICON_STYLE0, "Style:Line");
        AppendMenu(iconMenu, MF_STRING, ID_ICON_STYLE1, "Style:Small Arrow");
        AppendMenu(iconMenu, MF_STRING, ID_ICON_STYLE2, "Style:Arrow+Line");
        AppendMenu(iconMenu, MF_STRING, ID_ICON_STYLE3, "Style:Diagonal+Line");
        CheckMenuItem(iconMenu,IconStyle+2,MF_BYPOSITION|MF_CHECKED);
        if (!contextMenuDisable) {
          AppendMenu(iconMenu, MF_SEPARATOR , 0, 0);
          AppendMenu(iconMenu, MF_POPUP, (UINT)subMenu, "Main Menu");
        }
        TrackPopupMenu( iconMenu, TPM_RIGHTBUTTON |TPM_TOPALIGN |TPM_LEFTALIGN, here.x, here.y, 0, hwnd, NULL);
        PostMessage (hwnd, WM_NULL, 0, 0);
      }
      return FALSE;

    case WM_KEYDOWN:
      if (!gstart_time) {
        gstart_time = time (NULL);
        if (now)
           gstart_time -= tstep * (winW>>1);
      }

      switch ((int)wParam) {
        case VK_LEFT:
          if (graphmode || hairy) {
             UndrawCrosshair();
             CrosshairEnable = TRUE;
             if (ctl_key)
                NextEventAdjusted( &CrosshairTime, -1 );
             else
                CrosshairTime -= increment_step*60;
             if (CrosshairTime <= (window_left_time + HOURSECONDS)) {
                if (graphmode) {
                   gstart_time = CrosshairTime - HOURSECONDS;
                   if (usescrollbar) {
                      notnowtime = TRUE;
                      DoScrollBar(FALSE);
                   }
                   InvalidateRect(hwnd,NULL,TRUE);
                   UpdateWindow(hwnd);
                }
                else {
                   CrosshairEnable = FALSE;
                   CrosshairTime = window_left_time;
                }
             }
             else
                DrawCrosshair(CrosshairTime, NULL);  /* Draw new crosshair */
          }
          return FALSE;

        case VK_RIGHT:
          if (graphmode || hairy) {
             UndrawCrosshair();
             CrosshairEnable = TRUE;
             if (ctl_key)
                NextEventAdjusted( &CrosshairTime, 1 );
             else
                CrosshairTime += increment_step*60;
             if (CrosshairTime >= (window_left_time +winW*tstep -HOURSECONDS)) {
                if (graphmode) {
                   gstart_time = CrosshairTime - winW*tstep + HOURSECONDS;
                   if (usescrollbar) {
                      notnowtime = TRUE;
                      DoScrollBar(FALSE);
                   }
                   InvalidateRect(hwnd,NULL,TRUE);
                   UpdateWindow(hwnd);
                }
                else {
                   CrosshairEnable = FALSE;
                   CrosshairTime = window_left_time +winW*tstep;
                }
             }
             else
                DrawCrosshair(CrosshairTime, NULL);  /* Draw new crosshair */
          }
          return FALSE;

        case VK_CONTROL:
          ctl_key = TRUE;
          return FALSE;

        case VK_HOME:
          UndrawCrosshair(); /* Undraw current crosshair */
          CrosshairTime = gstart_time = time( NULL );
          if (now)
             gstart_time -= tstep * (winW>>1);
          if ((graphmode /*&& notnowtime*/) || CrosshairEnable) {
             InvalidateRect(hwnd,NULL,TRUE);
             UpdateWindow(hwnd);
          }
          if (graphmode&&usescrollbar) {
             notnowtime = FALSE;
             DoScrollBar(FALSE);
          }
          return FALSE;

        case VK_ESCAPE:
          if (CrosshairDrawn > 0) {
             UndrawCrosshair(); /* Undraw current crosshair */
             CrosshairEnable = FALSE;
          }
          else if (OnESCape == 1)       /* Minimize on ESCape */
            ShowWindow(hwnd, SW_MINIMIZE);

          else if (OnESCape == 2)       /* Exit on ESCape */
            PostMessage(hwnd, WM_CLOSE, 0, 0);

          return FALSE;
      }
      break;

    case WM_KEYUP:
      switch((int)wParam) {
        case VK_F1:
          WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Main graphics screen"));
          return FALSE;

        case VK_CONTROL:
          ctl_key = FALSE;
          return FALSE;
      }
      break;

//    case WM_RBUTTONDOWN:
    case WM_LBUTTONDOWN:
      {
        time_t mousetime = window_left_time;
        if (graphmode || overview || hairy) // NOT Clock mode
           mousetime += (LOWORD(lParam)*tstep);
        UndrawCrosshair();
        UseCrosshairPopup = 0;
        CrosshairTime = mousetime;
        open_popup(lParam, astro_info(mousetime));
        if (CrosshairEnable) {
          InvalidateRect( hwndMain, NULL, FALSE );
          UpdateWindow(hwndMain);
        }
      }
      return FALSE;

//    case WM_RBUTTONUP:
    case WM_LBUTTONUP:
      close_popup();
      UndrawCrosshair();
      UseCrosshairPopup = 1;
      if (CrosshairEnable) {
         InvalidateRect( hwndMain, NULL, FALSE );
         UpdateWindow(hwndMain);
      }
      return FALSE;

    case WM_HSCROLL:
      {
        unsigned short int scpos, sccode;

        if (!gstart_time) {
          gstart_time = time (NULL);
          if (now)
             gstart_time -= tstep * (winW>>1);
        }

        notnowtime = TRUE;
        sccode = wParam;
        scpos = HIWORD(wParam);       // Microsoft is not consistent!
             if (sccode == SB_LINELEFT)  gstart_time -= HOURSECONDS;
        else if (sccode == SB_LINERIGHT) gstart_time += HOURSECONDS;
        else if (sccode == SB_PAGELEFT)  gstart_time -= tstep*winW;
        else if (sccode == SB_PAGERIGHT) gstart_time += tstep*winW;
        else if (sccode == SB_ENDSCROLL) {
          InvalidateRect(hwnd,NULL,TRUE);
          DoScrollBar(FALSE);
        }
        else if (sccode == SB_THUMBTRACK) {
          char sbuf[80];
          struct tm *tm;
          POINT FAR mpos;
          gstart_time = scroll_bar_2_time_t(scpos);
          tm = tzlocaltime( &gstart_time );
          strftime( sbuf, sizeof(sbuf), "%Y-%m-%d ", tm );
          GetCursorPos(&mpos);
          ScreenToClient(hwndMain,&mpos);
          open_popup(MAKELONG(mpos.x,mpos.y), sbuf);
        }
        else if (sccode == SB_THUMBPOSITION) close_popup();// printf("ThumbPosition %d\n",scpos);
      }
      return FALSE;

    case WM_COMMAND:
      switch(LOWORD(wParam)) {
//        case 0://ID_COPY:
//            if(OpenClipboard(hwnd)) {
//            HGLOBAL hData=GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE,
//                                        sizeof(METAFILEPICT));
//            METAFILEPICT FAR * lpmfp=(METAFILEPICT FAR *)GlobalLock(hData);
//            HDC hdc;
//
//            EmptyClipboard();
//            lpmfp->mm=MM_TEXT;
//            lpmfp->xExt=winW;
//            lpmfp->yExt=winH;
//            hdc=CreateMetaFile(NULL);
//            DrawDC(hdc);
//            lpmfp->hMF=CloseMetaFile(hdc);
//            GlobalUnlock(hData);
//            SetClipboardData(CF_METAFILEPICT,hData);
//            CloseClipboard();
//        }
//        else
//            MessageBox(hwnd,"Windows Clipboard is not available","WinTide",MB_ICONINFORMATION);
//        break;

        case ID_ICON_STYLE0:
        case ID_ICON_STYLE1:
        case ID_ICON_STYLE2:
        case ID_ICON_STYLE3:
          if (IconStyle != (LOWORD(wParam) - ID_ICON_STYLE0)) {
            new_params = 1;
            IconStyle = LOWORD(wParam) - ID_ICON_STYLE0;
            ForceUpdateIcon = 1;
          }
          break;

        case ID_BELLS:
          ShipBells = !ShipBells;
          break;

        case ID_SHOW_HIDE:
          hold_restore = 1; /* Inhibit new_params */
          if (IsIconic(hwnd)) {
            RestoreMainWindow(hwnd);      // Restore main window original size
            SetForegroundWindow(hwnd);
          }
          else {
            ShowWindow(hwnd, SW_MINIMIZE);
            ShowWindow(hwnd, SW_HIDE);
          }
          hold_restore = 0;
          break;
/*
 * Window change sequencer
*/
        case ID_CHANGE_WINDOW:
          if (ChangingWindow) {
            section_restore();
            if (ChangingWindow != -1)   // not started minimized and going to normal
              PostMessage(hwnd, WM_COMMAND, ID_CHANGE_WINDOW_END, 0);
            else {
              ChangingWindow = 0;
              hold_restore   = 0;
            }
          }
          break;

        case ID_CHANGE_WINDOW_END:
          if (ChangingWindow) {
            hold_restore = 0;
//            if (ChangingWindow == -1) {  // Started minimized, going back minimized
//              ShowWindow(hwnd, SW_MINIMIZE);
//              ShowWindow(hwnd, SW_HIDE);
//              WindowWasMax = (ChangingWindow < -1);
//            }
//            else
              ShowWindow(hwnd, SW_SHOWMAXIMIZED);

            ChangingWindow = 0;
          }
          break;

        case ID_PRINT_GRAPH:
          print_graph = 1; // Fall through to print

        case ID_PRINT:
          {
            PRINTDLG pd;

            memset(&pd,0,sizeof pd);

            pd.lStructSize = sizeof pd;
            pd.hwndOwner = hwnd;
            pd.Flags = PD_RETURNDC|PD_NOPAGENUMS|PD_NOSELECTION;

            if(PrintDlg(&pd)) {
              DOCINFO di;
              TEXTMETRIC tm;
              SIZE sz, asz;
              int  LineSpace;
              int  clnPage;
              UINT cln;
              int  ipgMax;
              int  ipg;
              int  copies;
              UINT ilnLast;
              UINT ilnBase;
              UINT iln;
              HCURSOR hcurSave = SetCursor(LoadCursor(NULL, IDC_WAIT));

              sz.cx=GetDeviceCaps(pd.hDC,HORZRES);
              sz.cy=GetDeviceCaps(pd.hDC,VERTRES);

              di.cbSize=sizeof di;
              di.lpszDocName=location;
              di.lpszOutput=NULL;
              di.lpszDatatype=NULL;
              di.fwType=0;

//              GetTextMetrics(pd.hDC, &tm);
//              LineSpace = tm.tmHeight + tm.tmExternalLeading;
              if (print_graph) {
                 time_t ts = time (NULL);
                 char s[128];
                 GetTextMetrics(pd.hDC, &tm);
                 LineSpace = tm.tmHeight + tm.tmExternalLeading;
                 if (TRUE) {        // Maintain aspect ratio for printed page
                    float WinAspect, VpAspect;
                    asz.cx = sz.cx;
                    asz.cy = sz.cy - LineSpace*2;
                    WinAspect = (float)winW / (float)winH;
                    VpAspect  = (float)sz.cx/ (float)sz.cy;
                    if (WinAspect < VpAspect) asz.cx *= (WinAspect / VpAspect);
                    if (WinAspect > VpAspect) asz.cy *= (VpAspect / WinAspect);
                 }
                 strftime(s, sizeof(s), "%m/%d/%y %I:%m%p  ", tzlocaltime( &ts ));
                 sprintf(s+strlen(s),"   %s",custom_name);
                 StartDoc(pd.hDC,&di);
                 for (copies=pd.nCopies; copies > 0; copies--) {
                    StartPage(pd.hDC);
                    SetMapMode(pd.hDC,MM_ANISOTROPIC);
                    SetWindowOrgEx(pd.hDC,0,0,NULL);
                    SetWindowExtEx(pd.hDC,winW,winH,NULL);
                    SetViewportOrgEx(pd.hDC,0,0,NULL);
                    SetViewportExtEx(pd.hDC,sz.cx,sz.cy,NULL);
                    TextOut(pd.hDC, 0, 0, s, strlen(s));
                    SetViewportOrgEx(pd.hDC,0,LineSpace*2,NULL);
                    SetViewportExtEx(pd.hDC,asz.cx,asz.cy,NULL);
                    DrawDC(pd.hDC);
                    EndPage(pd.hDC);
                 }
              }

              else {
                 LOGFONT lf;
                 HFONT oldfont=NULL, pfont=NULL;
                 HWND  hwndCtl=GetDlgItem(hwndText,ID_TEXT);

                 memset(&lf, 0, sizeof(lf));
                 lf.lfHeight = -MulDiv(tFontSize, GetDeviceCaps(pd.hDC, LOGPIXELSY), 72);
                 lf.lfPitchAndFamily = FIXED_PITCH;
                 strcpy(lf.lfFaceName,tFontName);
                 pfont   = CreateFontIndirect(&lf); // Find printer font closest to text font
                 oldfont = SelectFont(pd.hDC,pfont);

                 GetTextMetrics(pd.hDC, &tm);
                 LineSpace = tm.tmHeight + tm.tmExternalLeading;
                 clnPage = sz.cy / LineSpace - 1;
                 cln = (UINT)SendMessage(hwndCtl, EM_GETLINECOUNT, 0, 0L);
                 ipgMax = cln / clnPage + 1;

                 StartDoc(pd.hDC,&di);
                 for (copies=pd.nCopies; copies > 0; copies--) {
                    for (ipg = 0; ipg < ipgMax ; ipg++) {
                       ilnBase = ipg * clnPage;
                       if (ipg == ipgMax-1) // Special case last page
                          ilnLast = (cln-1) % clnPage;
                       else
                          ilnLast = clnPage - 1;
                       StartPage(pd.hDC);
                       for (iln=0; iln <= ilnLast; iln++) {
                          char szLine[128];
                          int  cchLine;
                          szLine[0] = sizeof(szLine)-1;          // Maximum buffer size
                          szLine[1] = 0;
                          cchLine = (int)SendMessage(hwndCtl,
                                                     EM_GETLINE,
                                                     ilnBase + iln,
                                                     (DWORD)(LPSTR)szLine);
                          TextOut(pd.hDC, 0, iln*LineSpace, (LPSTR)szLine, cchLine);
                       }
                       EndPage(pd.hDC);
                    }
                 }
                 if (oldfont != NULL) SelectFont(pd.hDC,oldfont);
                 DeleteObject(pfont);
              }
              EndDoc(pd.hDC);

              if (pd.hDevMode != NULL)
                  GlobalFree(pd.hDevMode);
              if (pd.hDevNames != NULL)
                  GlobalFree(pd.hDevNames);
              DeleteDC(pd.hDC);
              SetCursor(hcurSave);    // Remove the hourglass
            }
          }
          print_graph = 0;
          break;

        case ID_CLOCK:
          {
            int wasgraph = graphmode || overview;
            int washairy = hairy;
            if (wasgraph || washairy) {
              section_save();
              new_params= TRUE;
              hairy    = FALSE;
              graphmode= FALSE;
              overview = FALSE;
              if (!section_restore())
                DoScrollBar(wasgraph&&usescrollbar);
              ChangeMainWindowMode(hwnd);
            }
          }
          break;

        case ID_HAIRY:
          {
            int wasgraph = graphmode || overview;
            int wasclock = !wasgraph && !hairy;
            if (wasgraph || wasclock) {
              section_save();
              new_params= TRUE;
              hairy    = TRUE;
              graphmode= FALSE;
              overview = FALSE;
              if (!section_restore())
                DoScrollBar(wasgraph&&usescrollbar);
              ChangeMainWindowMode(hwnd);
            }
          }
          break;

        case ID_GRAPH:
          {
            int wasgraph = graphmode || overview;
            int wasntgraph = overview || !graphmode;
            if (wasntgraph) {
              section_save();
              new_params= TRUE;
              hairy    = FALSE;
              graphmode= TRUE;
              overview = FALSE;
              if (!section_restore())
                DoScrollBar(!wasgraph&&usescrollbar);
              if (wasgraph && CrosshairEnable && CrosshairTime) { // Overview and had crosshair
                gstart_time = CrosshairTime - ((winW>>1)*tstep);  // Center on that time
                InvalidateRect(hwnd,NULL,TRUE);
              }
              ChangeMainWindowMode(hwnd);
            }
          }
          break;

        case ID_OVERVIEW:
          {
            int wasgraph = graphmode || overview;
            int wasntoverview = !overview;
            if (wasntoverview) {
              section_save();
              new_params= TRUE;
              hairy    = FALSE;
              graphmode= TRUE;
              overview = TRUE;
              if (!section_restore())
                DoScrollBar(!wasgraph&&usescrollbar);
              ChangeMainWindowMode(hwnd);
            }
          }
          break;

        case ID_CALENDAR:
        case ID_TIDES:
        case ID_INCREMENT:
          contextMenuDisable = 1;
          TextOption = LOWORD(wParam);
          DialogBox(g_hinst,"TextOpt",hwnd,TextOptionsDlg);
          contextMenuDisable = 0;
          SetFocus(hwndMain);
          break;

        case ID_LOCATION:
          KillTimer(hwnd,1);
          contextMenuDisable = 1;
          if (DialogBox(g_hinst,"EnterLocation",hwnd,LocationDlg)) {
             if (now)
                   faketime -= (long)tstep*(winW>>1);
             else  faketime  = time(NULL);
             next_ht = time(NULL);
             update_high_tide ();
             update_high_tide();
             NameWindow(hwnd, IDX_station_name);
             new_params = TRUE;
             DoScrollBar(FALSE);
          }
          SetTimer(hwnd,1,(UINT)TIMER_VALUE,NULL);
          contextMenuDisable = 0;
          ForceUpdateIcon = 1;
          break;

        case ID_PREFS:
          contextMenuDisable = 1;
          PrefsPropSheet( hwnd );
          contextMenuDisable = 0;
          ForceUpdateIcon = 1;
          break;

        case ID_MRU1:
        case ID_MRU2:
        case ID_MRU3:
        case ID_MRU4:
        case ID_MRU5:
        case ID_MRU6:
        case ID_MRU7:
        case ID_MRU8:
        case ID_MRU9:
          if (have_user_offsets &&
            (IDYES != MessageBox(hwnd,
                 "Custom station offsets are active and will\r\n"
                 "be disabled when the new station is loaded.\r\n\r\n"
                 "Load new station data?",
                 "Switch location", MB_ICONQUESTION|MB_YESNO)))
            break;
          Ihttimeoff = Ilttimeoff = 0;
          Ihlevelmult= Illevelmult= 1.0;
          Ihtleveloff= Iltleveloff= 0.0;
          have_user_offsets = 0;
          load_mru(LOWORD(wParam) - ID_MRU0);
          new_params      = TRUE;
          DoScrollBar(FALSE);
          ForceUpdateIcon = 1;
          break;

        case ID_CUSTOM:
          contextMenuDisable = 1;
          DialogBox(g_hinst,"Custom",hwnd,CustomDlg);
          contextMenuDisable = 0;
          ForceUpdateIcon = 1;
          break;

        case ID_SAVE_AS:
          init_OPENFILENAME(OFN_OVERWRITEPROMPT,
              "Save WXTide32 Configuration file");
          if (!GetSaveFileName(&ofn))
             return FALSE;
          have_new_custom = FALSE;
          write_config_file(ofn.lpstrFile, FALSE);
          strcpy(szConfigPath, ofn.lpstrFile);
          break;

        case ID_SAVE:
          write_config_file(szConfigPath, TRUE);
          break;

        case ID_LOAD:
          if (new_params) {
             if (IDYES != MessageBox(hwnd,
                "Some options have been changed and not saved.\r\n"
                "Load anyway and ignore changes?",
                "Options changed", MB_ICONEXCLAMATION|MB_YESNO))
                return FALSE;
          }
          new_params = FALSE;
          init_OPENFILENAME(OFN_FILEMUSTEXIST,
              "Load WXTide32 Configuration file");
          if (!GetOpenFileName(&ofn))
             return FALSE; // No file selected
          sprintf(szMessage,"\"%s\"",ofn.lpstrFile);
          ShellExecute(NULL, "open", szProgramName, szMessage,
             NULL, SW_SHOWNORMAL);  // Start me again with name of file as parameter string
//Fall through to exit this instance

        case ID_EXIT:
          PostMessage(hwnd,WM_CLOSE,0,0L);
          return FALSE;

        case ID_WEB:
          ShellExecute(NULL, "open", "http://wxtide32.com", NULL,
                       NULL, SW_SHOWNORMAL);  // Start default web browser at base web site
          break;

        case ID_EMAIL:
          ShellExecute(NULL, "open", "mailto:support@wxtide32.com", NULL,
                       NULL, SW_SHOWNORMAL);  // Start default mail client
          break;

        case ID_ABOUT:
          sprintf(szMessage,"WXTide32 version %s, %s\r\n"
                            "Copyright  1998-2007 Michael Hopper\r\n"
                            "\r\n"
                            "Web site: http://wxtide32.com\r\n"
                            "Email: support@wxtide32.com",
                            WXTIDE32VERSIONNUMBER, WXTIDE32VERSIONDATE);
          MessageBox(hwnd,szMessage,"About WXTide32",MB_OK|MB_ICONINFORMATION);
          break;

        case ID_HELP:
          WinHelp(hwnd,HelpFileName,HELP_FINDER,0);
          break;

        case ID_HELP_FAQ:
          WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"FAQ"));
          break;

        case ID_NOW:
        case ID_GOTO:
        case ID_GONOW:
          if (graphmode) {
            if ((notnowtime && LOWORD(wParam) == ID_GONOW) || LOWORD(wParam) == ID_NOW) {
              notnowtime = FALSE;
              CrosshairTime = gstart_time = time (NULL);
            }
            else {
              if (!DateTimePicker(hwnd))
                return FALSE;
              notnowtime = TRUE;
              CrosshairTime = gstart_time;// = time (NULL);
            }
            if (now)
               gstart_time -= tstep * (winW>>1);
            DoScrollBar(FALSE);
          }
          break;
      }
      InvalidateRect(hwnd,NULL,TRUE);
      UpdateWindow(hwnd);
      WinTideCheckMenus(hwnd);
      return FALSE;
  }
  return DefWindowProc(hwnd,msg,wParam,lParam);
}


int win_fprintf(FILE *stream,const char *fstring, ...) {
static int  nState=0;
static char szCaption[256];
static char szMessage[4096];
char        szOutput[1024];
int         nReturn;
va_list     marker;

  va_start(marker,fstring);
  if(stderr==stream) {
    nReturn=vsprintf(szOutput,fstring,marker);
    switch (nState) {
      case 0:
        if(0==lstrcmp("Tidelib Fatal Error:  ",szOutput)) {
          lstrcpy(szCaption,szOutput);
          nState++;
        }
        else {
          MessageBox(NULL,szOutput,"Fatal internal error",MB_ICONEXCLAMATION);
          PostQuitMessage(0);
          exit(-1);
//          nLenAccum+=strlen(szOutput+2);
//          if(szAccum)
//            szAccum=realloc(szAccum,nLenAccum);
//          else {
//            szAccum=malloc(nLenAccum);
//            szAccum[0]='\0';
//          }
//          lstrcat(szAccum,szOutput);
        }
        break;

      case 1:
        lstrcpy(szMessage,szOutput);
        nState++;
        break;

      case 2:
        lstrcat(szMessage,szOutput);
        MessageBox(NULL,szMessage,szCaption,MB_ICONEXCLAMATION);
        PostQuitMessage(0);
        exit(-1);
//        nState=0;
        break;

      default:
        //Oh shit!
        nState=0;
        break;
    }
  }
  else
    nReturn=vfprintf(stream,fstring,marker);
  return nReturn;
}

void win_assert_sub(void) {
  fprintf (stderr, "ASSERT failure - I know not where\n");
  PostQuitMessage(1);
  exit(-1);
}

MSG t_msg;
static int hold_text_window = 1;


int printf(const char *fstring, ...) {
char szOutput[4096], szTemp[4096], *pch;
int  nReturn, c;
va_list marker;
HWND hwndCtl;

  va_start(marker,fstring);
  nReturn=vsprintf(szTemp,fstring,marker);
  if (output_filename) {
     if (!textLFN) {
        if (!nowarn) {
           sprintf(szOutput, "Redirecting all text output to %s\r\n"
               "The -nowarn switch disables this message\r\n"
               "Press OK to continue.", output_filename);
           MessageBox(NULL, szOutput, "Text redirection",
           MB_TASKMODAL | MB_ICONINFORMATION);
        }
        textLFN=fopen( output_filename, "wt" );
     }
     fwrite( szTemp, strlen(szTemp), 1, textLFN );
     return(0);
  }
  pch = szOutput;
  for (c=0; c<nReturn; c++) // Parse the string to change \n to \r\n for windows
     if ((szTemp[c] == '\n') && ((c==0) || ((c>0) && (szTemp[c-1] != '\r')))) {
         *pch++ = '\r';
         *pch++ = '\n';
     }
     else
         *pch++ = szTemp[c];
  *pch = '\0';

  if(NULL==hwndText) {
     int cxscreen, cyscreen;
     hwndText=CreateDialog(t_hinst,"TextTides",NULL,TextTidesDlgProc);
     cxscreen = GetSystemMetrics(SM_CXSCREEN);
     cyscreen = GetSystemMetrics(SM_CYSCREEN);
     if ((WinDimText.left   >= 0) &&
         (WinDimText.right  < cxscreen) &&
         (WinDimText.top    >= 0) &&
         (WinDimText.bottom < cyscreen) &&
         (WinDimText.right  > WinDimText.left) &&
         (WinDimText.bottom > WinDimText.top ) &&
         (WinDimText.left|WinDimText.right|WinDimText.top|WinDimText.bottom) )
        MoveWindow(hwndText,WinDimText.left,WinDimText.top,
            WinDimText.right-WinDimText.left,WinDimText.bottom-WinDimText.top,TRUE);
     ShowWindow(hwndText,SW_SHOW);
     hold_text_window = 0;
  }

  if (tfont == NULL) {
    LOGFONT lf;
    HDC     hdcText = GetDC(hwndText);
    memset(&lf, 0, sizeof(lf));
    lf.lfHeight = -MulDiv(tFontSize, GetDeviceCaps(hdcText, LOGPIXELSY), 72);
    strcpy(lf.lfFaceName,tFontName);
    tfont=CreateFontIndirect(&lf);
    SendDlgItemMessage(hwndText,ID_TEXT,WM_SETFONT,(WPARAM)tfont, MAKELONG(TRUE,0));
    SendDlgItemMessage(hwndText,ID_TEXT,EM_SCROLLCARET,0,0);
    ReleaseDC(hwndText, hdcText);
  }

  hwndCtl=GetDlgItem(hwndText,ID_TEXT);
  SendMessage(hwndCtl,EM_SETSEL,(WPARAM)-1,(LPARAM)-1);
  SendMessage(hwndCtl,EM_REPLACESEL,0,(LPARAM)(LPSTR)szOutput);
  if (hwndMain) SetFocus(hwndMain);
  return nReturn;
}

BOOL CALLBACK TextTidesDlgProc(HWND hwndDlg,UINT t_msg,WPARAM wParam,LPARAM lParam)
{
RECT rc;
HWND hwndCtl;
SIZE sz;
HMENU smenu, tmenu, hmenu=GetMenu(hwndMain);

  switch(t_msg) {
    case WM_INITDIALOG:
      SendDlgItemMessage(hwndDlg,ID_TEXT,WM_SETFONT,
                        (WPARAM)GetStockFont(ANSI_FIXED_FONT),0L);
      SendDlgItemMessage(hwndDlg,ID_TEXT,EM_SETLIMITTEXT,(WPARAM)65534, 0L);
      WinTideCheckMenus(hwndMain);
      if(hmenu)
         EnableMenuItem(hmenu, ID_PRINT, MF_BYCOMMAND|MF_ENABLED);
//      if (hwndMain) SetFocus(hwndMain);
      smenu = GetSystemMenu(hwndDlg,FALSE);
      AppendMenu(smenu, MF_SEPARATOR, 0, (LPSTR) NULL);
      AppendMenu(smenu, MF_STRING, ID_PRINT, "Print...");
      return FALSE;

    case WM_CONTEXTMENU:
      tmenu = CreateMenu();//GetSystemMenu(hwndDlg,FALSE);
      AppendMenu(tmenu, MF_SEPARATOR, 0, (LPSTR) NULL);
      AppendMenu(tmenu, MF_STRING, ID_PRINT, "Print...");
      TrackPopupMenu( tmenu, TPM_RIGHTBUTTON |TPM_TOPALIGN |TPM_LEFTALIGN,
            LOWORD(lParam), HIWORD(lParam), 0, hwndText, NULL);
      return(FALSE);

    case WM_SIZE:
      hwndCtl=GetDlgItem(hwndDlg,ID_TEXT);
      GetWindowRect(hwndCtl,&rc);
      ScreenToClient(hwndDlg,(LPPOINT)&rc);
      sz.cx=LOWORD(lParam)-rc.left*2;
      sz.cy=HIWORD(lParam)-rc.top*2;
      MoveWindow(hwndCtl,rc.left,rc.top,sz.cx,sz.cy,TRUE);
      if (save_windows && !hold_text_window) {
         new_params = TRUE;
         WinTideCheckMenus(hwndMain);
      }
//      if (hwndMain) SetFocus(hwndMain);
      GetWindowRect(hwndText,&WinDimText);
      return TRUE;

    case WM_MOVE:
      if (save_windows && !hold_text_window) {
         new_params = TRUE;
         WinTideCheckMenus(hwndMain);
      }
      GetWindowRect(hwndText,&WinDimText);
      break;

    case WM_CLOSE:
      if(hmenu)
         EnableMenuItem(hmenu, ID_PRINT, MF_BYCOMMAND|MF_GRAYED);
      DestroyWindow(hwndDlg);
      hwndText = NULL;//mgh invalidate the window handle
      WinTideCheckMenus(hwndMain);
      if(done)
        PostQuitMessage(0);// Only do this if Started in text mode
      return TRUE;

  }
  return FALSE;
}